package de.ugoe.cs.eventbench.models;

import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import de.ugoe.cs.eventbench.data.Event;

/**
 * <p>
 * Implements a Deterministic Finite Automata (DFA) capable of random session
 * generation. It is a special case of a first-order Markov model, where the
 * transition probability is equally high for all following states.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class DeterministicFiniteAutomaton extends FirstOrderMarkovModel {

	/**
	 * <p>
	 * Id for object serialization.
	 * </p>
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * <p>
	 * Constructor. Creates a new DeterministicFiniteAutomaton.
	 * </p>
	 * 
	 * @param r
	 *            random number generator used by probabilistic methods of the
	 *            class
	 */
	public DeterministicFiniteAutomaton(Random r) {
		super(r);
	}

	/**
	 * <p>
	 * Calculates the proability of the next state. Each of the following states
	 * in the automaton is equally probable.
	 * </p>
	 * 
	 * @see de.ugoe.cs.eventbench.models.IStochasticProcess#getProbability(java.util.List,
	 *      de.ugoe.cs.eventbench.data.Event)
	 */
	@Override
	public double getProbability(List<? extends Event<?>> context,
			Event<?> symbol) {
		double result = 0.0d;

		List<Event<?>> contextCopy;
		if (context.size() >= trieOrder) {
			contextCopy = new LinkedList<Event<?>>(context.subList(
					context.size() - trieOrder + 1, context.size()));
		} else {
			contextCopy = new LinkedList<Event<?>>(context);
		}

		List<Event<?>> followers = trie.getFollowingSymbols(contextCopy);

		if (followers.size() != 0 && followers.contains(symbol)) {
			result = 1.0d / followers.size();
		}

		return result;
	}

}
