//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.htmlmonitor;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * represents an element of an HTML GUI. This can be a server, which is usually the root of a
 * GUI structure, a web page, or a tag within a web page.
 * </p>
 * 
 * @author Patrick Harms
 */
class HtmlPageElement {

    /**
     * path to the parent of this page element or null, if this is the root element
     */
    private String parentPath;
    
    /**
     * the name of the tag represented by this page element. May also be the name of the server
     * or the page.
     */
    private String tagName;
    
    /**
     * the id of the page element. May also be the name of the server including port number or the
     * URL of the page
     */
    private String id;
    
    /**
     * the title of the page if a page is represented through this element
     */
    private String title;
    
    /**
     * the index of this element regarding all elements with the same tag name in the list of 
     * children of the parent element
     */
    private Integer index;
    
    /**
     * the children of this element
     */
    private List<HtmlPageElement> children;

    /**
     * <p>
     * instantiates a new element representing a tag in an HTML page
     * </p>
     *
     * @param parentPath the path through the DOM to the parent node of the represented tag 
     * @param tagName    the name of the represented tag
     * @param id         the id of the tag in the DOM
     * @param index      the index of the represented tag regarding all tags with the same tag name
     *                   in the list of children of the parent tag
     */
    HtmlPageElement(String parentPath, String tagName, String id, Integer index) {
        this.parentPath = parentPath;
        this.tagName = tagName;
        this.id = id;
        this.index = index;
    }

    /**
     * <p>
     * instantiates a new element representing an HTML page
     * </p>
     *
     * @param parentPath the path through the DOM to the parent node of the represented tag which is
     *                   usually a server
     * @param tagName    the name of the represented tag which is the path of the web page
     * @param title      the title of the web page
     * @param id         the id of the web page which is its path
     * @param index      usually 0
     */
    HtmlPageElement(String parentPath, String tagName, String id, String title, Integer index) {
        this(parentPath, tagName, id, index);
        this.title = title;
    }

    /**
     * <p>
     * returns the name of the tag represented by this page element. May also be the name of the
     * server or the page.
     * </p>
     * 
     * @return the tagName
     */
    String getTagName() {
        return tagName;
    }

    /**
     * <p>
     * returns the id of the page element. May also be the name of the server including port
     * number or the URL of the page
     * </p>
     * 
     * @return the id
     */
    String getId() {
        return id;
    }

    /**
     * <p>
     * returns the title of the page if a page is represented through this element
     * </p>
     * 
     * @return the title
     */
    String getTitle() {
        return title;
    }

    /**
     * <p>
     * returns the index of this element regarding all elements with the same tag name in the list
     * of children of the parent element
     * </p>
     * 
     * @return the index
     */
    Integer getIndex() {
        return index;
    }

    /**
     * <p>
     * returns the children of this element if any, null else
     * </p>
     * 
     * @return the children
     */
    List<HtmlPageElement> getChildren() {
        return children;
    }


    /**
     * adds a child to this element
     * 
     * @param child the child to be added
     */
    void addChild(HtmlPageElement child) {
        if (child != null) {
            if (children == null) {
                children = new ArrayList<HtmlPageElement>();
            }
            
            children.add(child);
        }
    }

    /**
     * <p>
     * returns the path to the parent of this page element or null, if this is the root element
     * </p>
     *
     * @return as described
     */
    String getParentPath() {
        return parentPath;
    }

    /**
     * <p>
     * calculates and returns the path to the represented page element through the DOM. Includes
     * the parent path if any. The represented element is included in the path with its tag name
     * and its id if it has one, or its index.
     * </p>
     *
     * @return as described
     */
    String getPath() {
        StringBuffer result = new StringBuffer();
        if (parentPath != null) {
            result.append(parentPath);
        }

        result.append("/");
        result.append(tagName);
        
        if ((id != null) && (!"".equals(id))) {
            result.append("(id=");
            result.append(id);
            result.append(")");
        }
        else {
            result.append("[");
            result.append(index);
            result.append("]");
        }
        
        return result.toString();
    }

}
