package de.ugoe.cs.eventbench.models;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;

import de.ugoe.cs.eventbench.data.Event;

/**
 * <p>
 * This interface defines the functionalities provided by stochastic processes.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public interface IStochasticProcess extends Serializable {

	/**
	 * <p>
	 * Returns the probability, that the next event is {@code symbol} given the
	 * last events are {@code context}. The last element of {@code context} is
	 * the most recent in the history, the first element is the oldest.
	 * </p>
	 * 
	 * @param context
	 *            recently observed symbols
	 * @param symbol
	 *            event for which the probability is calculated
	 * @return probabilty the {@code symbol} is the next event, given the last
	 *         events {@code context}
	 */
	double getProbability(List<? extends Event<?>> context, Event<?> symbol);

	/**
	 * <p>
	 * Generates a random sequence of events. The sequence starts with
	 * {@link Event#STARTEVENT} and finishes with {@link Event#ENDEVENT}.
	 * </p>
	 * 
	 * @return randomly generated sequence
	 */
	public List<? extends Event<?>> randomSequence();

	/**
	 * <p>
	 * Generates all sequences of a given length are possible, i.e., have a
	 * positive probability.<br>
	 * All states are used as possible starting states.
	 * </p>
	 * 
	 * @param length
	 *            length of the generated sequences
	 * @return generated sequences
	 * @see #generateSequences(int, boolean)
	 */
	public Collection<List<? extends Event<?>>> generateSequences(int length);

	/**
	 * <p>
	 * Generates all sequences of given length that can are possible, i.e., have
	 * positive probability.<br>
	 * If {@code fromStart==true}, all generated sequences start in
	 * {@link Event#STARTEVENT}. Otherwise this method is the same as
	 * {@link #generateSequences(int)}.
	 * </p>
	 * 
	 * @param length
	 *            length of the generated sequences
	 * @param fromStart
	 *            if true, all generated sequences start with
	 *            {@link Event#STARTEVENT}
	 * @return generated sequences
	 */
	public Collection<List<? extends Event<?>>> generateSequences(int length,
			boolean fromStart);

	/**
	 * <p>
	 * Generates all sequences starting with {@link Event#STARTEVENT} and
	 * finishing with {@link Event#ENDEVENT} of a given length. It is possible
	 * that no such sequence exists with the defined length and the returned set
	 * is empty. If {@code length} is less than 2 the returned set is always
	 * empty.
	 * </p>
	 * 
	 * @param length
	 * @return generated sequences
	 */
	public Collection<List<? extends Event<?>>> generateValidSequences(int length);

	/**
	 * <p>
	 * Returns the number of states known by the stochastic process, i.e., the
	 * size of its alphabet.
	 * </p>
	 * 
	 * @return number of states
	 */
	public int getNumStates();

	/**
	 * <p>
	 * Returns a string representation of all known states.
	 * </p>
	 * 
	 * @return string representation for all known states
	 */
	public String[] getStateStrings();

	/**
	 * <p>
	 * Returns all states known by the stochastic process, i.e., its
	 * {@link Event}s.
	 * </p>
	 * 
	 * @return events known by the process
	 */
	public Collection<? extends Event<?>> getEvents();

}