package de.ugoe.cs.autoquest.usability.evaluation.rule.evaluator;

import static de.ugoe.cs.autoquest.usability.tasktree.filter.EventTypeFilter.TEXT_INPUT;
import static de.ugoe.cs.autoquest.usability.util.TextInputUtil.aggregateEnteredTextFromTextInputs;
import static de.ugoe.cs.autoquest.usability.util.TextInputUtil.characterIsLetterOrDigitPredicate;

import com.google.common.base.CharMatcher;
import com.google.common.base.Optional;
import com.google.common.collect.Multiset;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTree;
import de.ugoe.cs.autoquest.usability.evaluation.result.UsabilityDefectSeverityLevel;
import de.ugoe.cs.autoquest.usability.evaluation.rule.set.UsabilityRule;
import de.ugoe.cs.autoquest.usability.tasktree.filter.FilterStatistic;
import de.ugoe.cs.autoquest.usability.tasktree.filter.IterativeDFSFilterStrategy;
import de.ugoe.cs.autoquest.usability.tasktree.filter.TaskTreeFilter;

public class NoLetterOrDigitTextInputsEvaluator extends RuleEvaluator {

    public NoLetterOrDigitTextInputsEvaluator(UsabilityRule evaluatedUsabilityRule, ITaskTree taskTree) {
        super(evaluatedUsabilityRule, taskTree);
    }

    @Override
    protected FilterStatistic nodesUnderEvaluation(ITaskTree taskTree) {
        Optional<FilterStatistic> cachedNodes = loadFromCache(TEXT_INPUT);
        return cachedNodes.isPresent() ? cachedNodes.get() : cacheAndReturnNodes(taskTree, TEXT_INPUT);
    }
    
    @Override
    protected FilterStatistic extractNodesFromTaskTree(ITaskTree taskTree) {
        return new TaskTreeFilter(new IterativeDFSFilterStrategy())
            .filterByEventType(TEXT_INPUT).from(taskTree);
    }

    @Override
    protected float calculateEvaluationMetric() {
        Multiset<String> enteredTextFragments = aggregateEnteredTextFromTextInputs(this.filteredNodes.nodesMatchedFilter());
        int allCharactersCount = 0;
        int noLetterOrDigitCount = 0;
        for(String textFragment : enteredTextFragments.elementSet()) {
            int occurencesOfTextFragment = enteredTextFragments.count(textFragment);
            allCharactersCount += CharMatcher.ANY.countIn(textFragment) * occurencesOfTextFragment;
            noLetterOrDigitCount += CharMatcher.forPredicate(characterIsLetterOrDigitPredicate()).countIn(textFragment) * occurencesOfTextFragment;
        }
        return allCharactersCount != 0 ? (float) noLetterOrDigitCount / (float) allCharactersCount : 0;
    }

    @Override
    protected Optional<UsabilityDefectSeverityLevel> determineSeverityLevel(float evaluationMetric) {
        Optional<UsabilityDefectSeverityLevel> recommendationSeverityLevel = Optional.absent();
        if (evaluationMetric > 0.1) // every 10th sign
        {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.HIGH);
        }
        else if (evaluationMetric > 0.05) // every 20th sign
        {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.MEDIUM);
        }
        else if (evaluationMetric > 0.02) // every 50th sign
        {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.LOW);
        }
        else if (evaluationMetric > 0.01) // every 100th sign
        {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.INFO);
        }
        return recommendationSeverityLevel;
    }
 
}
