
package de.ugoe.cs.autoquest.usability.evaluation.rule.evaluator;

import static de.ugoe.cs.autoquest.usability.evaluation.rule.set.RulesetFactory.textInputUsabiliyRuleset;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.absent;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.highRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.infoRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.lowRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.mediumRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.present;
import static org.fest.assertions.api.Assertions.assertThat;

import java.util.EnumSet;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTree;
import de.ugoe.cs.autoquest.usability.evaluation.result.UsabilityDefect;
import de.ugoe.cs.autoquest.usability.evaluation.rule.set.UsabilityRule;
import de.ugoe.cs.autoquest.usability.tasktree.filter.FilterStatisticCache;

public class TextInputRatioEvaluatorTest extends AbstractUsabilityEvaluationTC {

    private UsabilityRule ruleUnderTest;

    @Before
    public void initRuleUnderTest() {
        EnumSet<? extends UsabilityRule> ruleset =
            textInputUsabiliyRuleset().evaluationRules();
        this.ruleUnderTest = Iterables.find(ruleset, new Predicate<UsabilityRule>() {

            public boolean apply(UsabilityRule usabilityRule) {
                return usabilityRule.ruleIdentifier().equals("TEXT_FIELD_INPUT_RATIO");
            }

        });
    }

    @After
    public void clearFilterStatisticCache() {
        FilterStatisticCache.instance().clear();
    }

    @Test
    public void should_return_no_recommendation() {
        // Given
        String spec =
            "Sequence {" + "  Interaction {}" + "  TextInput (a) {}" + "  Interaction {}"
                + "  Interaction {}" + "  TextInput (c) {}" + "  Interaction {}"
                + "  Interaction {}" + "  Interaction {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputRatioEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(absent());
    }

    @Test
    public void should_return_recommendation_with_info_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  Interaction {}" + "  TextInput (a) {}" + "  Interaction {}"
                + "  Interaction {}" + "  TextInput (c) {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputRatioEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(present()).has(infoRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_low_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  Interaction {}" + "  TextInput (a) {}" + "  TextInput (b) {}"
                + "  Interaction {}" + "  TextInput (c) {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputRatioEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(present()).has(lowRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_medium_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a) {}" + "  TextInput (b) {}" + "  Interaction {}"
                + "  TextInput (c) {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputRatioEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(present()).has(mediumRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_high_severity_level() {
        // Given
        String spec = "TextInput (bla) {}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputRatioEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(present()).has(highRecommendationSeverityLevel());
    }
}
