
package de.ugoe.cs.autoquest.usability.evaluation.rule.evaluator;

import static de.ugoe.cs.autoquest.usability.evaluation.rule.set.RulesetFactory.textInputUsabiliyRuleset;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.absent;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.highRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.infoRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.lowRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.mediumRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.present;
import static org.fest.assertions.api.Assertions.assertThat;

import java.util.EnumSet;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTree;
import de.ugoe.cs.autoquest.usability.evaluation.result.UsabilityDefect;
import de.ugoe.cs.autoquest.usability.evaluation.rule.set.UsabilityRule;
import de.ugoe.cs.autoquest.usability.tasktree.filter.FilterStatisticCache;

public class TextInputEntryRepetitionsEvaluatorTest extends AbstractUsabilityEvaluationTC {

    private UsabilityRule ruleUnderTest;

    @Before
    public void initRuleUnderTest() {
        EnumSet<? extends UsabilityRule> ruleset =
            textInputUsabiliyRuleset().evaluationRules();
        this.ruleUnderTest = Iterables.find(ruleset, new Predicate<UsabilityRule>() {

            public boolean apply(UsabilityRule usabilityRule) {
                return usabilityRule.ruleIdentifier().equals("TEXT_FIELD_INPUT_REPETITIONS");
            }

        });
    }

    @After
    public void clearFilterStatisticCache() {
        FilterStatisticCache.instance().clear();
    }

    @Test
    public void should_return_no_recommendation() {
        // Given
        String spec = "Sequence {" + "  TextInput () {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputEntryRepetitionsEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(absent());
    }

    @Test
    public void should_return_recommendation_with_info_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a b c) {}" + "  Sequence {" + "    TextInput (a) {}"
                + "    TextInput (d) {}" + "    TextInput (e) {}" + "  }" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputEntryRepetitionsEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(present()).has(infoRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_low_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a b c) {}" + "  Sequence {" + "    TextInput (a) {}"
                + "    TextInput (b) {}" + "    TextInput (c) {}" + "  }" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputEntryRepetitionsEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(present()).has(lowRecommendationSeverityLevel());    }

    @Test
    public void should_return_recommendation_with_medium_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a b c d e f g h i j k l m) {}" + "  Sequence {"
                + "    TextInput (a) {}" + "    TextInput (b) {}" + "    TextInput (c) {}"
                + "    TextInput (d) {}" + "  }" + "  Iteration {" + "    TextInput (e) {}" + "  }"
                + "  TextInput (a) {}" + "  Selection {" + "    TextInput (b) {}"
                + "    TextInput (c) {}" + "    TextInput (d) {}" + "    TextInput (e) {}" + "  }"
                + "  Sequence {" + "    TextInput (a) {}" + "    Sequence {"
                + "      TextInput (b) {}" + "      TextInput (c) {}" + "      TextInput (d) {}"
                + "      TextInput (e) {}" + "    }" + "  }" + "  TextInput (f) {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputEntryRepetitionsEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(present()).has(mediumRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_high_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a b c) {}" + "  Sequence {" + "    TextInput (a) {}"
                + "    TextInput (b) {}" + "    TextInput (c) {}" + "    TextInput (a) {}" + "  }"
                + "  Iteration {" + "    TextInput (a) {}" + "  }" + "  TextInput (a) {}"
                + "  Selection {" + "    TextInput (b c) {}" + "    TextInput (a) {}"
                + "    TextInput (a c) {}" + "    TextInput (b a) {}" + "  }" + "  Sequence {"
                + "    TextInput (b c) {}" + "    Sequence {" + "      TextInput (d a c) {}"
                + "      TextInput (b b b a) {}" + "      TextInput (a a c c) {}"
                + "      TextInput (b b a) {}" + "    }" + "  }" + "  TextInput (d) {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation =
            new TextInputEntryRepetitionsEvaluator(this.ruleUnderTest, taskTree).evaluationResult();
        // Then
        assertThat(recommendation).is(present()).has(highRecommendationSeverityLevel());
    }
}
