package de.ugoe.cs.autoquest.usability.evaluation.rule.evaluator;

import static de.ugoe.cs.autoquest.usability.tasktree.filter.EventTypeFilter.TEXT_INPUT;
import static de.ugoe.cs.autoquest.usability.util.TextInputUtil.aggregateEnteredTextFromTextInputs;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Multiset;
import com.google.common.collect.Multisets;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTree;
import de.ugoe.cs.autoquest.usability.evaluation.result.UsabilityDefectSeverityLevel;
import de.ugoe.cs.autoquest.usability.evaluation.rule.set.UsabilityRule;
import de.ugoe.cs.autoquest.usability.tasktree.filter.FilterStatistic;
import de.ugoe.cs.autoquest.usability.tasktree.filter.IterativeDFSFilterStrategy;
import de.ugoe.cs.autoquest.usability.tasktree.filter.TaskTreeFilter;

public class TextInputEntryRepetitionsEvaluator extends RuleEvaluator {

    public TextInputEntryRepetitionsEvaluator(UsabilityRule evaluatedUsabilityRule, ITaskTree taskTree) {
        super(evaluatedUsabilityRule, taskTree);
    }

    @Override
    protected FilterStatistic nodesUnderEvaluation(ITaskTree taskTree) {
        Optional<FilterStatistic> cachedNodes = loadFromCache(TEXT_INPUT);
        return cachedNodes.isPresent() ? cachedNodes.get() : cacheAndReturnNodes(taskTree, TEXT_INPUT);
    }
    
    @Override
    protected FilterStatistic extractNodesFromTaskTree(ITaskTree taskTree) {
        return new TaskTreeFilter(new IterativeDFSFilterStrategy())
            .filterByEventType(TEXT_INPUT).from(taskTree);
    }

    @Override
    protected float calculateEvaluationMetric() {
        Multiset<String> enteredTextFragments = aggregateEnteredTextFromTextInputs(this.filteredNodes.nodesMatchedFilter());
        Multiset<String> orderedTextFragmentsWithMultipleOccurences = onlyTextFragmentsWithMultipleOccurences(enteredTextFragments);
        if(orderedTextFragmentsWithMultipleOccurences.isEmpty()) return 0;
        String wordWithHighestRepetitionInTextFragments = orderedTextFragmentsWithMultipleOccurences.iterator().next();
        int numberOfRepeatedWords = orderedTextFragmentsWithMultipleOccurences.entrySet().size();
        int maxRepetitions = orderedTextFragmentsWithMultipleOccurences.count(wordWithHighestRepetitionInTextFragments);
        return Math.max(numberOfRepeatedWords, maxRepetitions);
    }
    
    private Multiset<String> onlyTextFragmentsWithMultipleOccurences(final Multiset<String> allTextInputs) {
        return Multisets.copyHighestCountFirst(Multisets.filter(allTextInputs, new Predicate<String>() {
            
            @Override
            public boolean apply(String word) {
                return allTextInputs.count(word) > 1;
            }
            
        }));
    }

    @Override
    protected Optional<UsabilityDefectSeverityLevel> determineSeverityLevel(float evaluationMetric) {
        Optional<UsabilityDefectSeverityLevel> recommendationSeverityLevel = Optional.absent();
        if (evaluationMetric > 10) {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.HIGH);
        }
        else if (evaluationMetric > 4) {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.MEDIUM);
        }
        else if (evaluationMetric > 2) {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.LOW);
        }
        else if (evaluationMetric > 1) {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.INFO);
        }
        return recommendationSeverityLevel;
    }
    
}
