//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.mfc;

import de.ugoe.cs.autoquest.eventcore.guimodel.GUIElementTree;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec;
import de.ugoe.cs.autoquest.plugin.mfc.guimodel.MFCGUIElementSpec;

/**
 * <p>
 * Message handler for {@code WM_CREATE} messages. The handler maintains the {@link GUIElementTree}.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class HandlerCreate extends MessageHandler {

    /**
     * <p>
     * Constructor. Creates a new HandlerCreate.
     * </p>
     * 
     * @param guiElementTree the tree of GUI element specifications to be created and adapted during
     *                   parsing
     */
    public HandlerCreate(GUIElementTree<Long> guiElementTree) {
        super(guiElementTree);
    }

    /**
     * <p>
     * Name of the created GUI element.
     * </p>
     */
    private String guiElementName;

    /**
     * <p>
     * HWND of the created GUI element.
     * </p>
     */
    private long hwnd;

    /**
     * <p>
     * HWND of the created GUI elements's parent.
     * </p>
     */
    private long parentHwnd;

    /**
     * <p>
     * Resource Id of the created element.
     * </p>
     */
    private int resourceId;

    /**
     * <p>
     * GUI element class of the created element.
     * </p>
     */
    private String className;

    /**
     * <p>
     * Modality of the created GUI element.
     * </p>
     */
    private boolean isModal;

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.plugin.mfc.MessageHandler#onEndElement()
     */
    @Override
    public void onEndElement() {
        if (hwnd != 0) {
        	IGUIElementSpec spec = new MFCGUIElementSpec(hwnd, guiElementName, resourceId, className, isModal);
            super.getGUIElementTree().add(hwnd, parentHwnd, spec);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.plugin.mfc.MessageHandler#onParameter(java.lang.String ,
     * java.lang.String)
     */
    @Override
    public void onParameter(String name, String value) {
        if (name.equals("window.hwnd")) {
            hwnd = Long.parseLong(value);
        }
        else if (name.equals("window.name")) {
            guiElementName = value;
        }
        else if (name.equals("window.parent.hwnd")) {
            parentHwnd = Long.parseLong(value);
        }
        else if (name.equals("window.resourceId")) {
            resourceId = Integer.parseInt(value);
        }
        else if (name.equals("window.class")) {
            if (value.startsWith("Afx:")) {
                className = "Afx:";
            }
            else {
                className = value;
            }
        }
        else if (name.equals("window.ismodal")) {
            if (value.equals("true") || value.equals("1")) {
                isModal = true;
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.plugin.mfc.MessageHandler#onStartElement()
     */
    @Override
    public void onStartElement() {
        guiElementName = "";
        hwnd = 0;
        parentHwnd = 0;
        resourceId = 0;
        className = "";
        isModal = false;
    }

}
