//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.htmlmonitor;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * represents an element of an HTML GUI. This can be a server, which is usually the root of a
 * GUI structure, a web page, or a tag within a web page.
 * </p>
 * 
 * @author Patrick Harms
 */
abstract class HtmlGUIElement {

    /**
     * the children of this element
     */
    private List<HtmlGUIElement> children;
    
    /**
     * the id of this GUI element
     */
    private String id;

    /**
     * the parent GUI element, null if none is present
     */
    private HtmlGUIElement parent;

    /**
     * <p>
     * instantiates a new GUI element
     * </p>
     * 
     * @param parentId the parent GUI element, null if none is present
     */
    protected HtmlGUIElement(String id, HtmlGUIElement parent) {
        if (id == null) {
            throw new IllegalArgumentException("id must not be null");
        }
        
        this.id = id;
        this.parent = parent;
    }

    /**
     * <p>
     * returns the children of this element if any, null else
     * </p>
     * 
     * @return the children
     */
    List<HtmlGUIElement> getChildren() {
        return children;
    }


    /**
     * adds a child to this element. Merges children, for which the equal method returns true
     * 
     * @param child the child to be added
     */
    void addChild(HtmlGUIElement newChild) {
        if (newChild != null) {
            if (children == null) {
                children = new ArrayList<HtmlGUIElement>();
            }
            
            boolean added = false;
            for (HtmlGUIElement child : children) {
                if (child.equals(newChild)) {
                    if (newChild.getChildren() != null) {
                        for (HtmlGUIElement subchild : newChild.getChildren()) {
                            child.addChild(subchild);
                        }
                    }
                    
                    added = true;
                    break;
                }
            }
            
            if (!added) {
                children.add(newChild);
            }
        }
    }
    
    /**
     * returns the id of the GUI element.
     * 
     * @return the id of the GUI element
     */
    String getId() {
        return id;
    }

    /**
     * <p>
     * returns the id of the parent GUI element, if any
     * </p>
     *
     * @return the id of the parent GUI element, or null, if this element does not have a parent
     */
    String getParentId() {
        if (parent == null) {
            return null;
        }
        else {
            return parent.getId();
        }
    }

    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public abstract boolean equals(HtmlGUIElement obj);

}
