//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.htmlmonitor;

/**
 * <p>
 * convenience interface for all components making up the HTML monitor. All components implement
 * this interface to ensure homogeneity throughout them. A component must first be initialized
 * ({@link #init()}), then started ({@link #start()}) and finally stopped ({@link #stop()}).
 * </p>
 * 
 * @author Patrick Harms
 */
interface HtmlMonitorComponent {

    /**
     * initializes a component, i.e. it does everything needed to prepare starting the component
     * 
     * @throws IllegalStateException thrown if the method was already called before but
     *                               {@link #stop()} wasn't called yet
     * @throws HtmlMonitorException  thrown if the initialization fails, e.g. because needed infos
     *                               are missing
     */
    void init() throws IllegalStateException, HtmlMonitorException;

    /**
     * starts a component
     * 
     * @throws IllegalStateException thrown if {@link #init()} wasn't called yet of if the component
     *                               is already started through a preceding call to this method
     * @throws HtmlMonitorException  thrown if the startup fails, e.g. because needed infos
     *                               are missing
     */
    void start() throws IllegalStateException, HtmlMonitorException;

    /**
     * stops a component and cleans up any derivate. In the following, {@link #init()} must be
     * callable again. If the component is not initialized or started, nothing must happen. If the
     * component is initialized but not started, the initialization is revoked.
     */
    void stop();
}
