package de.ugoe.cs.autoquest.tasktrees.nodeequality;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;

/**
 * <p>
 * This class is capable of comparing any task tree node which is not a selection with a
 * selection. This is needed, because selections may contain exactly that node. Therefore, if
 * this node is selected out of a selection the selection is equal to the node itself. 
 * The rule returns lexically equal, it the selection contains a lexically equal node. The same
 * applies for syntactical and semantical equality.
 * </p>

 * @author Patrick Harms
 */
public class NodeAndSelectionComparisonRule implements NodeComparisonRule {
    
    /** the rule manager for internally comparing task tree nodes */
    private NodeEqualityRuleManager mRuleManager;

    /**
     * <p>
     * simple constructor to provide the rule with the node equality rule manager to be able
     * to perform comparisons of the children of provided task tree nodes
     * </p>
     * 
     * @param ruleManager the rule manager for comparing task tree nodes
     */
    NodeAndSelectionComparisonRule(NodeEqualityRuleManager ruleManager) {
        super();
        mRuleManager = ruleManager;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.tasktree.nodeequality.NodeEqualityRule#apply(TaskTreeNode, TaskTreeNode)
     */
    @Override
    public NodeEquality compare(ITaskTreeNode node1, ITaskTreeNode node2) {
        ISelection selection = null;
        ITaskTreeNode node = null;
        
        if (node1 instanceof ISelection) {
            if (node2 instanceof ISelection) {
                // the rule is not responsible for two iterations
                return null;
            }
            
            selection = (ISelection) node1;
            node = node2;
        }
        else if (node2 instanceof ISelection) {
            if (node1 instanceof ISelection) {
                // the rule is not responsible for two iterations
                return null;
            }
            
            selection = (ISelection) node2;
            node = node1;
        }
        else {
            return null;
        }

        // now, that we found the iteration and the node, lets compare the child of the iteration
        // with the node.
        if (selection.getChildren().size() < 1) {
            return null;
        }

        NodeEquality mostConcreteNodeEquality = null;
        
        for (ITaskTreeNode child : selection.getChildren()) {
            NodeEquality nodeEquality = mRuleManager.applyRules(child, node);
            
            if (nodeEquality != NodeEquality.UNEQUAL) {
                if (mostConcreteNodeEquality == null) {
                    mostConcreteNodeEquality = nodeEquality;
                }
                else {
                    mostConcreteNodeEquality =
                        mostConcreteNodeEquality.getCommonDenominator(nodeEquality);
                }
            }
        }
        
        // although the subtask may be identical to the node, we can not return identical, as
        // the selection is not identical to the node, but at most lexically equal
        if (mostConcreteNodeEquality == NodeEquality.IDENTICAL) {
            return NodeEquality.LEXICALLY_EQUAL;
        }
        else {
            return mostConcreteNodeEquality;
        }

    }
}
