//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.evaluation.rule.evaluator;

import static de.ugoe.cs.autoquest.usability.tasktree.filter.EventTypeFilter.TEXT_INPUT;
import static de.ugoe.cs.autoquest.usability.util.TextInputUtil.aggregateEnteredTextFromTextInputs;
import static java.lang.String.format;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Multiset;
import com.google.common.collect.Multisets;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTree;
import de.ugoe.cs.autoquest.usability.evaluation.result.UsabilityDefectSeverityLevel;
import de.ugoe.cs.autoquest.usability.evaluation.rule.set.UsabilityRule;
import de.ugoe.cs.autoquest.usability.tasktree.filter.FilterStatistic;
import de.ugoe.cs.autoquest.usability.tasktree.filter.IterativeDFSFilterStrategy;
import de.ugoe.cs.autoquest.usability.tasktree.filter.TaskTreeFilter;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Alexander Deicke
 */
public class TextInputEntryRepetitionsEvaluator extends RuleEvaluator {

    public TextInputEntryRepetitionsEvaluator(UsabilityRule evaluatedUsabilityRule,
                                              ITaskTree taskTree)
    {
        super(evaluatedUsabilityRule, taskTree);
    }

    @Override
    protected FilterStatistic nodesUnderEvaluation(ITaskTree taskTree) {
        Optional<FilterStatistic> cachedNodes = loadFromCache(TEXT_INPUT);
        return cachedNodes.isPresent() ? cachedNodes.get() : cacheAndReturnNodes(taskTree,
                                                                                 TEXT_INPUT);
    }

    @Override
    protected FilterStatistic extractNodesFromTaskTree(ITaskTree taskTree) {
        return new TaskTreeFilter(new IterativeDFSFilterStrategy()).filterByEventType(TEXT_INPUT)
            .from(taskTree);
    }

    @Override
    protected float calculateEvaluationMetric() {
        Multiset<String> enteredTextFragments =
            aggregateEnteredTextFromTextInputs(this.filteredNodes.nodesMatchedFilter());
        Multiset<String> orderedTextFragmentsWithMultipleOccurences =
            onlyTextFragmentsWithMultipleOccurences(enteredTextFragments);
        if (orderedTextFragmentsWithMultipleOccurences.isEmpty())
            return 0;
        String wordWithHighestRepetitionInTextFragments =
            orderedTextFragmentsWithMultipleOccurences.iterator().next();
        int numberOfRepeatedWords = orderedTextFragmentsWithMultipleOccurences.entrySet().size();
        int maxRepetitions =
            orderedTextFragmentsWithMultipleOccurences
                .count(wordWithHighestRepetitionInTextFragments);
        storeEvaluationMetricForDefectDescription("textRepetitionRatio",
                                                  format("textRepetitionRatio %s repeated tokens, up to %s repetitions per token",
                                                         numberOfRepeatedWords, maxRepetitions));
        return Math.max(numberOfRepeatedWords, maxRepetitions);
    }

    private Multiset<String> onlyTextFragmentsWithMultipleOccurences(final Multiset<String> allTextInputs)
    {
        return Multisets.copyHighestCountFirst(Multisets.filter(allTextInputs,
                                                                new Predicate<String>() {

                                                                    @Override
                                                                    public boolean apply(String word)
                                                                    {
                                                                        return allTextInputs
                                                                            .count(word) > 1;
                                                                    }

                                                                }));
    }

    @Override
    protected Optional<UsabilityDefectSeverityLevel> determineSeverityLevel(float evaluationMetric)
    {
        Optional<UsabilityDefectSeverityLevel> recommendationSeverityLevel = Optional.absent();
        if (evaluationMetric > 10) {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.HIGH);
        }
        else if (evaluationMetric > 4) {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.MEDIUM);
        }
        else if (evaluationMetric > 2) {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.LOW);
        }
        else if (evaluationMetric > 1) {
            recommendationSeverityLevel = Optional.of(UsabilityDefectSeverityLevel.INFO);
        }
        return recommendationSeverityLevel;
    }

}
