//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.temporalrelation;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;

/**
 * <p>
 * a temporal relationship rule is able to detected temporal relationships between the child nodes
 * of the parent node provided to the {@link #apply(ITaskTreeNode, boolean)} method. A rule
 * creates temporal relationships between the child nodes, i.e. substructures in the task tree, if
 * it detects a temporal relationship and it can be sure that it is complete. Incomplete but
 * detected temporal relationships may occur, if there can be more children expected to be added
 * to the provided parent node. This could be the case during parsing a interaction log file of
 * a GUI.
 * </p>
 * 
 * @author Patrick Harms
 */
interface TemporalRelationshipRule {

  /**
   * <p>
   * applies the rule to the given parent node. The finalize parameter is used to command the rule
   * to finish rule applications, in the case it is known that no further data will be available. 
   * </p>
   * <p>
   * The returned rule application result is null, if the rule can not be applied, i.e. it does not
   * detect a temporal relationship. It returns a rule application result with a status
   * {@link RuleApplicationStatus#RULE_APPLICATION_FINISHED} if the rule was applied. The result
   * contains all newly created parent nodes. It returns a rule application result with status
   * {@link RuleApplicationStatus#RULE_APPLICATION_FEASIBLE} if the rule would be applicable if
   * further children would be available in the parent node. This status MUST not be returned if 
   * the finalize parameter is true. In this case the rule must be applied or not. 
   * </p>
   * 
   * @param parent   the parent node with the children to apply the rule on
   * @param finalize true, if the rule shall not expect further children to come and that it
   *                 should therefore be applied in any case
   *                    
   * @return the rule application result as described.
   */
  RuleApplicationResult apply(ITaskTreeNode parent,
                              boolean       finalize);
  
}
