//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.nodeequality;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;

/**
 * <p>
 * This class is capable of comparing Iterations. Iterations equal at distinct levels
 * in distinct situations. The following table shows the results of the comparison for the
 * specific situations (the parameters are commutative). In any other situation, the comparison
 * returns <code>NodeEquality.UNEQUAL</code>:
 * </p>
 * 
 * <table border="1">
 *   <tr>
 *     <th>iteration 1</th>
 *     <th>iteration 2</th>
 *     <th>comparison result</th>
 *   </tr>
 *   <tr>
 *     <td>any iteration</td>
 *     <td>any iteration with a child that is lexically equal to the child of iteration 1</td>
 *     <td><code>NodeEquality.LEXICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>any iteration</td>
 *     <td>any iteration with a child that is syntactically equal to the child of iteration 1</td>
 *     <td><code>NodeEquality.SYNTACTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>any iteration</td>
 *     <td>any iteration with a child that is semantically equal to the child of iteration 1</td>
 *     <td><code>NodeEquality.SEMANTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>an iteration with a selection of syntactically equal children</td>
 *     <td>an iteration with a child that is syntactically equal to the children of the child
 *     selection of iteration 1</td>
 *     <td><code>NodeEquality.SYNTACTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>an iteration with a selection of syntactically equal children</td>
 *     <td>an iteration with a selection of syntactically equal children that are all syntactically
 *     equal to the selection of children of iteration 1</td>
 *     <td><code>NodeEquality.SYNTACTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>an iteration with a selection of semantically equal children</td>
 *     <td>an iteration with a child that is semantically equal to the children of the child
 *     selection of iteration 1</td>
 *     <td><code>NodeEquality.SEMANTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>an iteration with a selection of semantically equal children</td>
 *     <td>an iteration with a selection of semantically equal children that are all semantically
 *     equal to the selection of children of iteration 1</td>
 *     <td><code>NodeEquality.SEMANTICALLY_EQUAL</code></td>
 *   </tr>
 * </table>
 * 
 * @version $Revision: $ $Date: 19.02.2012$
 * @author 2012, last modified by $Author: patrick$
 */
public class IterationComparisonRule implements NodeComparisonRule {
    
    /** the rule manager for internally comparing task tree nodes */
    private NodeEqualityRuleManager mRuleManager;

    /**
     * <p>
     * simple constructor to provide the rule with the node equality rule manager to be able
     * to perform comparisons of the children of provided task tree nodes
     * </p>
     * 
     * @param ruleManager the rule manager for comparing task tree nodes
     */
    IterationComparisonRule(NodeEqualityRuleManager ruleManager) {
        super();
        mRuleManager = ruleManager;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.tasktree.nodeequality.NodeEqualityRule#apply(TaskTreeNode, TaskTreeNode)
     */
    @Override
    public NodeEquality compare(ITaskTreeNode node1, ITaskTreeNode node2) {
        if ((!(node1 instanceof IIteration)) || (!(node2 instanceof IIteration))) {
            return null;
        }

        if (node1 == node2) {
            return NodeEquality.IDENTICAL;
        }

        // if both iterations do not have children, they are equal although this doesn't make sense
        if ((node1.getChildren().size() == 0) && (node2.getChildren().size() == 0)) {
            return NodeEquality.LEXICALLY_EQUAL;
        }
        else if ((node1.getChildren().size() == 0) || (node2.getChildren().size() == 0)) {
            return NodeEquality.UNEQUAL;
        }

        ITaskTreeNode child1 = node1.getChildren().get(0);
        ITaskTreeNode child2 = node2.getChildren().get(0);

        // iterations may have 3 different structures.
        // 1. they have one child, which is the iterated one
        // 2. they have a sequence of children, which is iterated
        // 3. they have a selection of different iterated variants (usually the variants are
        // semantically equal)
        //
        // the permutations of the three variants in combination must be checked

        // check if both nodes are the same variants of iterations and if their children are equal.
        // This condition matches, if both iterations are the same variants of iteration. I.e. three
        // combinations of the permutation are handled herewith.
        NodeEquality nodeEquality = mRuleManager.applyRules(child1, child2);

        if (nodeEquality.isAtLeast(NodeEquality.SEMANTICALLY_EQUAL)) {
            // prevent, that identical is returned, because the iterations itself are not identical
            // although the iterated tasks are
            if (nodeEquality == NodeEquality.IDENTICAL) {
                return NodeEquality.LEXICALLY_EQUAL;
            }
            else {
                return nodeEquality;
            }
        }

        // compare one iteration with a single node as a child and another one with a selection of
        // semantically equal nodes
        return selectionChildrenSemanticallyEqualNode(child1, child2);
        
        // all other combinations (i.e. sequence with single child and sequence with selection)
        // can not match
    }

    /**
     * <p>
     * compares two task tree nodes. One of them must be a selection, the other one can be any task
     * tree node. The method returns a node equality that is not <code>NodeEquality.UNEQUAL</code>
     * if the other node is at least semantically equal to the children of the selection. It
     * returns more concrete equalities, if the equality between the other node and the children
     * of the selection is more concrete.
     * </p> 
     * 
     * @param taskTreeNode  the first task tree node to compare
     * @param taskTreeNode2 the second task tree node to compare
     * 
     * @return as described
     */
    private NodeEquality selectionChildrenSemanticallyEqualNode(ITaskTreeNode taskTreeNode,
                                                                ITaskTreeNode taskTreeNode2)
    {
        ISelection selection = null;
        ITaskTreeNode node = null;
        if (taskTreeNode instanceof ISelection) {
            selection = (ISelection) taskTreeNode;
            node = taskTreeNode2;
        }
        else if (taskTreeNode2 instanceof ISelection) {
            selection = (ISelection) taskTreeNode2;
            node = taskTreeNode;
        }
        else {
            return NodeEquality.UNEQUAL;
        }

        // Iterations, where one has a selection and the other one not can at most be syntactically
        // equal but not identical
        NodeEquality commonDenominatorForAllComparisons = NodeEquality.SYNTACTICALLY_EQUAL;

        for (ITaskTreeNode child : selection.getChildren()) {
            NodeEquality nodeEquality = mRuleManager.applyRules(node, child);

            if ((nodeEquality == null) || (nodeEquality == NodeEquality.UNEQUAL))
            {
                return NodeEquality.UNEQUAL;
            }
            
            commonDenominatorForAllComparisons =
                commonDenominatorForAllComparisons.getCommonDenominator(nodeEquality);
        }

        return commonDenominatorForAllComparisons;
    }

}
