//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.temporalrelation;

import java.io.PrintStream;
import java.text.DecimalFormat;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Patrick Harms
 */
public class StopWatch {
    
    /**
     * 
     */
    private Map<String, List<Long>> mWatches = new HashMap<String, List<Long>>();

    /**
     *
     *
     * @param id
     */
    public void start(String id) {
        List<Long> timeStamps = mWatches.get(id);
        
        if (timeStamps == null) {
            timeStamps = new LinkedList<Long>();
            mWatches.put(id, timeStamps);
        }
        
        if (timeStamps.size() % 2 == 0) {
            timeStamps.add(System.currentTimeMillis());
        }
        else {
            throw new IllegalStateException("watch with id " + id + " already running");
        }
    }
    
    /**
     *
     *
     * @param id
     */
    public void stop(String id) {
        List<Long> timeStamps = mWatches.get(id);
       
        if (timeStamps == null) {
            throw new IllegalArgumentException("watch with id " + id + " does not exist");
        }
       
        if (timeStamps.size() % 2 == 0) {
            throw new IllegalStateException("watch with id " + id + " already stopped");
        }
        else {
            timeStamps.add(System.currentTimeMillis());
        }
    }
    
    /**
     *
     *
     * @param id
     */
    public long getDuration(String id) {
        List<Long> timeStamps = mWatches.get(id);
       
        if (timeStamps == null) {
            throw new IllegalArgumentException("watch with id " + id + " does not exist");
        }
       
        if (timeStamps.size() % 2 != 0) {
            stop(id);
        }
        
        long result = 0;
        for (long timeStamp : timeStamps) {
            if (result >= 0) {
                result -= timeStamp;
            }
            else {
                result += timeStamp;
            }
        }
        
        return result;
    }

    /**
     *
     * @param out
     */
    public void dumpStatistics(PrintStream out) {
        if (mWatches.size() <= 0) {
            throw new IllegalStateException("no watches registered that could be dumped");
        }
        
        Map<String, Long> durations = new HashMap<String, Long>();
        
        // get durations
        for (String id : mWatches.keySet()) {
            durations.put(id, getDuration(id));
        }
        
        // sort by duration
        List<String> sortedIds = new LinkedList<String>();
        int maxIdLength = 0;
        
        for (Map.Entry<String, Long> entry : durations.entrySet()) {
            boolean added = false;
            for (int i = 0; i < sortedIds.size(); i++) {
                if (durations.get(sortedIds.get(i)) >= entry.getValue()) {
                    sortedIds.add(i, entry.getKey());
                    added = true;
                    break;
                }
            }
           
            if (!added) {
                sortedIds.add(entry.getKey());
            }
            
            maxIdLength = Math.max(maxIdLength, entry.getKey().length());
        }
        
        // get longest duration and check whether it spans all other entries
        String id = sortedIds.get(sortedIds.size() - 1);
        List<Long> timeStamps = mWatches.get(id);
        long firstTimeStamp = timeStamps.get(0);
        long lastTimeStamp = timeStamps.get(timeStamps.size() - 1);
        long longestDuration = durations.get(id);
        boolean longestDurationCoversOthers = true;
        
        for (Map.Entry<String, List<Long>> watch : mWatches.entrySet()) {
            if ((watch.getValue().get(0) < firstTimeStamp) ||
                (watch.getValue().get(watch.getValue().size() - 1) > lastTimeStamp))
            {
                longestDurationCoversOthers = false;
                break;
            }
        }
        
        // no finally start the dumping
        out.println();
        out.println("Watch Statistics");
        out.println("================");

        for (String sortedId : sortedIds) {
            out.print(sortedId);
            
            for (int i = sortedId.length(); i <= maxIdLength; i++) {
                out.print(' ');
            }
            
            out.print(':');
            out.print(' ');
            
            out.print(durations.get(sortedId));
            out.print(" ms");
            
            if (longestDurationCoversOthers) {
                out.print(" (");
                out.print(DecimalFormat.getPercentInstance().format
                              ((double) durations.get(sortedId) / longestDuration));
                out.print(" of overall duration)");
            }
            out.println();
        }
        
        out.println();
    }
}
