//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.nodeequality;

import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;

/**
 * <p>
 * this node comparison rule is capable of comparing selections. If both selections do not have
 * children, they are treated as identical. If they have children, each child of both selections
 * is compared to each child of the respective other selection. The resulting equality is the most
 * concrete one of all these comparisons. I.e. if all children are at least lexically equal, then
 * the selections are lexically equal. If all children are at least syntactically equal, then the
 * selections are syntactically equal. If all children are at least semantically equal, then the
 * selections are semantically equal. If only one of the selections has children, then the
 * selections are unequal.
 * </p>
 * 
 * @version $Revision: $ $Date: 19.02.2012$
 * @author 2012, last modified by $Author: patrick$
 */
public class SelectionComparisonRule implements NodeComparisonRule {

    /** the rule manager for internally comparing task tree nodes */
    private NodeEqualityRuleManager mRuleManager;

    /**
     * <p>
     * simple constructor to provide the rule with the node equality rule manager to be able
     * to perform comparisons of the children of provided task tree nodes
     * </p>
     * 
     * @param ruleManager the rule manager for comparing task tree nodes
     */
    SelectionComparisonRule(NodeEqualityRuleManager ruleManager) {
        super();
        mRuleManager = ruleManager;
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#isApplicable(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean isApplicable(ITaskTreeNode node1, ITaskTreeNode node2) {
        return (node1 instanceof ISelection) && (node2 instanceof ISelection);
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areLexicallyEqual(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean areLexicallyEqual(ITaskTreeNode node1, ITaskTreeNode node2) {
        NodeEquality equality = getEquality(node1, node2, NodeEquality.LEXICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(NodeEquality.LEXICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSyntacticallyEqual(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean areSyntacticallyEqual(ITaskTreeNode node1, ITaskTreeNode node2) {
        NodeEquality equality = getEquality(node1, node2, NodeEquality.SYNTACTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(NodeEquality.SYNTACTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSemanticallyEqual(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean areSemanticallyEqual(ITaskTreeNode node1, ITaskTreeNode node2) {
        NodeEquality equality = getEquality(node1, node2, NodeEquality.SEMANTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(NodeEquality.SEMANTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#compare(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public NodeEquality compare(ITaskTreeNode node1, ITaskTreeNode node2) {
        return getEquality(node1, node2, null);
    }

    /**
     * 
     */
    private NodeEquality getEquality(ITaskTreeNode node1,
                                     ITaskTreeNode node2,
                                     NodeEquality  requiredEqualityLevel)
    {
        List<ITaskTreeNode> children1 = node1.getChildren();
        List<ITaskTreeNode> children2 = node2.getChildren();

        // if both selections do not have children, they are lexically equal. If only one of them
        // has children, they are unequal.
        if ((children1.size() == 0) && (children2.size() == 0)) {
            return NodeEquality.LEXICALLY_EQUAL;
        }
        else if ((children1.size() == 0) || (children2.size() == 0)) {
            return NodeEquality.UNEQUAL;
        }

        NodeEquality selectionEquality;

        if (requiredEqualityLevel == null) {
            // calculate the common equality level for all children of both selections.
            // do it in both directions to ensure commutative comparison
            selectionEquality = getCommonEqualityLevel(children1, children2);
            if (selectionEquality != NodeEquality.UNEQUAL) {
                return selectionEquality.getCommonDenominator
                    (getCommonEqualityLevel(children2, children1));
            }
            else {
                return NodeEquality.UNEQUAL;
            }
        }
        else {
            // we are searching for a specific equality
            if (checkEqualityLevel(children1, children2, requiredEqualityLevel) &&
                checkEqualityLevel(children2, children1, requiredEqualityLevel))
            {
                return requiredEqualityLevel;
            }
            else {
                return NodeEquality.UNEQUAL;
            }
        }
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param children1
     * @param children2
     * @param requiredEqualityLevel
     */
    private NodeEquality getCommonEqualityLevel(List<ITaskTreeNode> children1,
                                                List<ITaskTreeNode> children2)
    {
        NodeEquality listEquality = NodeEquality.LEXICALLY_EQUAL;
        
        NodeEquality childEquality;
        NodeEquality currentEquality;
        for (ITaskTreeNode child1 : children1) {
            childEquality = null;
            for (ITaskTreeNode child2 : children2) {
                currentEquality = callRuleManager(child1, child2, null);
                if ((currentEquality != null) && (currentEquality != NodeEquality.UNEQUAL)) {
                    if (childEquality == null) {
                        childEquality = currentEquality;
                    }
                    else {
                        childEquality = childEquality.getCommonDenominator(currentEquality);
                    }
                    
                    if (childEquality == NodeEquality.SEMANTICALLY_EQUAL) {
                        // as we calculate only the common denominator, we can break up here for
                        // the current child. We will not improve the denominator anymore
                        break;
                    }
                }
            }
            
            if (childEquality == null) {
                // we did not find any child in the second list, that is equal to the searched
                // child
                return NodeEquality.UNEQUAL;
            }
            else {
                listEquality = listEquality.getCommonDenominator(childEquality);
            }
        }

        return listEquality;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param children1
     * @param children2
     * @param requiredEqualityLevel
     */
    private boolean checkEqualityLevel(List<ITaskTreeNode> children1,
                                       List<ITaskTreeNode> children2,
                                       NodeEquality        requiredEqualityLevel)
    {
        NodeEquality childEquality;
        NodeEquality currentEquality;
        for (ITaskTreeNode child1 : children1) {
            childEquality = null;
            for (ITaskTreeNode child2 : children2) {
                currentEquality = callRuleManager(child1, child2, requiredEqualityLevel);
                if ((currentEquality != null) && (currentEquality.isAtLeast(requiredEqualityLevel)))
                {
                    // we found at least one equal child with sufficient equality in the
                    // second list. So be can break up for this child.
                    childEquality = currentEquality;
                    break;
                }
            }
            
            if (childEquality == null) {
                // we did not find any child in the second list, that is equal to the searched
                // child
                return false;
            }
        }

        // for all children, we found an equality 
        return true;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param child1
     * @param child2
     * @param requiredEqualityLevel
     * @return
     */
    private NodeEquality callRuleManager(ITaskTreeNode child1,
                                         ITaskTreeNode child2,
                                         NodeEquality  requiredEqualityLevel)
    {
        if (requiredEqualityLevel == null) {
            return mRuleManager.compare(child1, child2);
        }
        else if (mRuleManager.areAtLeastEqual(child1, child2, requiredEqualityLevel)) {
            return requiredEqualityLevel;
        }
        else {
            return NodeEquality.UNEQUAL;
        }
    }

}
