//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.nodeequality;

import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;

/**
 * <p>
 * This rule is capable of comparing sequences. If both sequences do not have children, they are
 * treated as lexically equal. Sequences are lexically equal, if they have the same number and
 * order of lexically equal children. The rule can not decide, if two sequences are syntactically
 * or semantically equal.
 * </p>
 * 
 * @version $Revision: $ $Date: 19.02.2012$
 * @author 2012, last modified by $Author: patrick$
 */
public class SequenceComparisonRule implements NodeComparisonRule {

    /** the rule manager for internally comparing task tree nodes */
    private NodeEqualityRuleManager mRuleManager;

    /**
     * <p>
     * simple constructor to provide the rule with the node equality rule manager to be able
     * to perform comparisons of the children of provided task tree nodes
     * </p>
     * 
     * @param ruleManager the rule manager for comparing task tree nodes
     */
    SequenceComparisonRule(NodeEqualityRuleManager ruleManager) {
        super();
        mRuleManager = ruleManager;
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#isApplicable(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean isApplicable(ITaskTreeNode node1, ITaskTreeNode node2) {
        return (node1 instanceof ISequence) && (node2 instanceof ISequence);
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areLexicallyEqual(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean areLexicallyEqual(ITaskTreeNode node1, ITaskTreeNode node2) {
        NodeEquality equality = getEquality(node1, node2, NodeEquality.LEXICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(NodeEquality.LEXICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSyntacticallyEqual(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean areSyntacticallyEqual(ITaskTreeNode node1, ITaskTreeNode node2) {
        NodeEquality equality = getEquality(node1, node2, NodeEquality.SYNTACTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(NodeEquality.SYNTACTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSemanticallyEqual(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean areSemanticallyEqual(ITaskTreeNode node1, ITaskTreeNode node2) {
        NodeEquality equality = getEquality(node1, node2, NodeEquality.SEMANTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(NodeEquality.SEMANTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#compare(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public NodeEquality compare(ITaskTreeNode node1, ITaskTreeNode node2) {
        return getEquality(node1, node2, null);
    }

    /**
     * 
     */
    private NodeEquality getEquality(ITaskTreeNode node1,
                                     ITaskTreeNode node2,
                                     NodeEquality  requiredEqualityLevel)
    {
        List<ITaskTreeNode> children1 = node1.getChildren();
        List<ITaskTreeNode> children2 = node2.getChildren();

        // if both sequences do not have children, they are equal although this doesn't make sense
        if ((children1.size() == 0) && (children2.size() == 0)) {
            return NodeEquality.LEXICALLY_EQUAL;
        }

        if (children1.size() != children2.size()) {
            return NodeEquality.UNEQUAL;
        }

        NodeEquality resultingEquality = NodeEquality.LEXICALLY_EQUAL;
        for (int i = 0; i < children1.size(); i++) {
            ITaskTreeNode child1 = children1.get(i);
            ITaskTreeNode child2 = children2.get(i);

            NodeEquality nodeEquality = callRuleManager(child1, child2, requiredEqualityLevel);

            if ((nodeEquality == null) || (nodeEquality == NodeEquality.UNEQUAL)) {
                return NodeEquality.UNEQUAL;
            }
            
            resultingEquality = resultingEquality.getCommonDenominator(nodeEquality);
        }

        return resultingEquality;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param child1
     * @param child2
     * @param requiredEqualityLevel
     * @return
     */
    private NodeEquality callRuleManager(ITaskTreeNode child1,
                                         ITaskTreeNode child2,
                                         NodeEquality  requiredEqualityLevel)
    {
        if (requiredEqualityLevel == null) {
            return mRuleManager.compare(child1, child2);
        }
        else if (mRuleManager.areAtLeastEqual(child1, child2, requiredEqualityLevel)) {
            return requiredEqualityLevel;
        }
        else {
            return NodeEquality.UNEQUAL;
        }
    }
}
