//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.temporalrelation;

import java.util.HashMap;

import de.ugoe.cs.autoquest.tasktrees.nodeequality.NodeEquality;
import de.ugoe.cs.autoquest.tasktrees.nodeequality.NodeEqualityRuleManager;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;
import de.ugoe.cs.autoquest.usageprofiles.SymbolComparator;
import de.ugoe.cs.util.StopWatch;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Patrick Harms
 */
public class TaskTreeNodeComparator implements SymbolComparator<ITaskTreeNode> {
    
    /**
     * <p>
     * the node equality manager needed for comparing task tree nodes with each other
     * </p>
     */
    private NodeEqualityRuleManager nodeEqualityRuleManager;

    /**
     * <p>
     * the minimal node equality two identified sublists need to have to consider them as equal
     * and to create an iteration for
     * </p>
     */
    private NodeEquality minimalNodeEquality;

    private Comparer comparer;

    private Comparer lexicalComparer;

    private StopWatch stopWatch = new StopWatch();
    
    private HashMap<Long, Boolean> equalityBuffer = new HashMap<Long, Boolean>();

    private HashMap<Long, Boolean> lexicalEqualityBuffer;

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param nodeEqualityRuleManager
     * @param minimalNodeEquality
     */
    public TaskTreeNodeComparator(NodeEqualityRuleManager nodeEqualityRuleManager,
                                  NodeEquality            minimalNodeEquality)
    {
        super();
        this.nodeEqualityRuleManager = nodeEqualityRuleManager;
        this.minimalNodeEquality = minimalNodeEquality;
        
        if (minimalNodeEquality == NodeEquality.LEXICALLY_EQUAL) {
            comparer = new LexicalComparer();
        }
        else if (minimalNodeEquality == NodeEquality.SYNTACTICALLY_EQUAL) {
            comparer = new SyntacticalComparer();
        }
        else if (minimalNodeEquality == NodeEquality.SEMANTICALLY_EQUAL) {
            comparer = new SemanticalComparer();
        }
        else {
            comparer = new DefaultComparer();
        }
        
        if (minimalNodeEquality == NodeEquality.LEXICALLY_EQUAL) {
            lexicalComparer = comparer;
            lexicalEqualityBuffer = equalityBuffer;
        }
        else {
            lexicalComparer = new LexicalComparer();
            lexicalEqualityBuffer = new HashMap<Long, Boolean>();
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.temporalrelation.SymbolComparator#equals(java.lang.Object, java.lang.Object)
     */
    @Override
    public boolean equals(ITaskTreeNode symbol1, ITaskTreeNode symbol2) {
        //String id = "compare " + symbol1.getClass().getSimpleName() + " " +
        //    symbol2.getClass().getSimpleName();
        //String id = "compare";
        //stopWatch.start(id);
        
        Boolean result;
        
        if (symbol1 != symbol2) {
            long key = ((long) System.identityHashCode(symbol1)) << 32;
            key += System.identityHashCode(symbol2);
            
            result = equalityBuffer.get(key);
            
            if (result == null) {
                result = comparer.compare(symbol1, symbol2);
                equalityBuffer.put(key, result);
            }
        }
        else {
            result = true;
        }
        //stopWatch.stop(id);
        
        /*boolean result2 = nodeEqualityRuleManager.areAtLeastEqual(symbol1, symbol2, minimalNodeEquality);
        if (result != result2) {
            throw new IllegalStateException("implementation error");
        }*/
        
        return result;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @return
     */
    StopWatch getStopWatch() {
        return stopWatch;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param node1
     * @param node2
     * @return
     */
    boolean areLexicallyEqual(ITaskTreeNode symbol1, ITaskTreeNode symbol2) {
        Boolean result;
        
        if (symbol1 != symbol2) {
            long key = ((long) System.identityHashCode(symbol1)) << 32;
            key += System.identityHashCode(symbol2);
            
            result = lexicalEqualityBuffer.get(key);
            
            if (result == null) {
                result = lexicalComparer.compare(symbol1, symbol2);
                lexicalEqualityBuffer.put(key, result);
            }
        }
        else {
            result = true;
        }
        
        return result;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @return
     */
    NodeEquality getConsideredNodeEquality() {
        return minimalNodeEquality;
    }

    /**
     * 
     */
    private interface Comparer {
        /**
         * 
         */
        boolean compare(ITaskTreeNode node1, ITaskTreeNode node2);
    }

    /**
     * 
     */
    private class LexicalComparer implements Comparer {
        
        /**
         * 
         */
        public boolean compare(ITaskTreeNode node1, ITaskTreeNode node2) {
            return nodeEqualityRuleManager.areLexicallyEqual(node1, node2);
        }
    }

    /**
     * 
     */
    private class SyntacticalComparer implements Comparer {
        
        /**
         * 
         */
        public boolean compare(ITaskTreeNode node1, ITaskTreeNode node2) {
            return nodeEqualityRuleManager.areSyntacticallyEqual(node1, node2);
        }
    }

    /**
     * 
     */
    private class SemanticalComparer implements Comparer {
        
        /**
         * 
         */
        public boolean compare(ITaskTreeNode node1, ITaskTreeNode node2) {
            return nodeEqualityRuleManager.areSemanticallyEqual(node1, node2);
        }
    }

    /**
     * 
     */
    private class DefaultComparer implements Comparer {
        
        /**
         * 
         */
        public boolean compare(ITaskTreeNode node1, ITaskTreeNode node2) {
            return nodeEqualityRuleManager.areAtLeastEqual(node1, node2, minimalNodeEquality);
        }
    }
}
