//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.nodeequality;

import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;

/**
 * <p>
 * This class is capable of comparing any task tree node which is not an iteration with an
 * iteration. This is needed, because iterations may iterate exactly that node. In this
 * case, the iteration would be equal to that node if it was executed exactly once. The rule
 * returns lexically equal, it the child of the iteration is lexically equal to the node
 * or if the child of the iteration is a selection and this selections contains a lexically equal
 * node. The same applies for syntactical and semantical equality.
 * </p>

 * @author Patrick Harms
 */
public class NodeAndIterationComparisonRule implements NodeComparisonRule {
    
    /** the rule manager for internally comparing task tree nodes */
    private NodeEqualityRuleManager mRuleManager;

    /**
     * <p>
     * simple constructor to provide the rule with the node equality rule manager to be able
     * to perform comparisons of the children of provided task tree nodes
     * </p>
     * 
     * @param ruleManager the rule manager for comparing task tree nodes
     */
    NodeAndIterationComparisonRule(NodeEqualityRuleManager ruleManager) {
        super();
        mRuleManager = ruleManager;
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#isApplicable(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean isApplicable(ITaskTreeNode node1, ITaskTreeNode node2) {
        return ((node1 instanceof IIteration) && (!(node2 instanceof IIteration))) ||
               ((node2 instanceof IIteration) && (!(node1 instanceof IIteration)));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areLexicallyEqual(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean areLexicallyEqual(ITaskTreeNode node1, ITaskTreeNode node2) {
        NodeEquality equality = getEquality(node1, node2, NodeEquality.LEXICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(NodeEquality.LEXICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSyntacticallyEqual(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean areSyntacticallyEqual(ITaskTreeNode node1, ITaskTreeNode node2) {
        NodeEquality equality = getEquality(node1, node2, NodeEquality.SYNTACTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(NodeEquality.SYNTACTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSemanticallyEqual(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public boolean areSemanticallyEqual(ITaskTreeNode node1, ITaskTreeNode node2) {
        NodeEquality equality = getEquality(node1, node2, NodeEquality.SEMANTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(NodeEquality.SEMANTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#compare(ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public NodeEquality compare(ITaskTreeNode node1, ITaskTreeNode node2) {
        return getEquality(node1, node2, null);
    }

    /**
     * 
     */
    private NodeEquality getEquality(ITaskTreeNode node1,
                                     ITaskTreeNode node2,
                                     NodeEquality  requiredEqualityLevel)
    {
        IIteration iteration = null;
        ITaskTreeNode node = null;
        
        if (node1 instanceof IIteration) {
            if (node2 instanceof IIteration) {
                // the rule is not responsible for two iterations
                return null;
            }
            
            iteration = (IIteration) node1;
            node = node2;
        }
        else if (node2 instanceof IIteration) {
            if (node1 instanceof IIteration) {
                // the rule is not responsible for two iterations
                return null;
            }
            
            iteration = (IIteration) node2;
            node = node1;
        }
        else {
            return null;
        }

        List<ITaskTreeNode> children = iteration.getChildren();
        
        // now, that we found the iteration and the node, lets compare the child of the iteration
        // with the node.
        if (children.size() < 1) {
            return null;
        }

        NodeEquality nodeEquality = callRuleManager(children.get(0), node, requiredEqualityLevel);

        // although the subtask may be identical to the node, we can not return identical, as
        // the iteration is not identical to the node, but at most lexically equal
        if (nodeEquality == NodeEquality.IDENTICAL) {
            return NodeEquality.LEXICALLY_EQUAL;
        }
        else {
            return nodeEquality;
        }

    }
    
    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param child1
     * @param child2
     * @param requiredEqualityLevel
     * @return
     */
    private NodeEquality callRuleManager(ITaskTreeNode child1,
                                         ITaskTreeNode child2,
                                         NodeEquality  requiredEqualityLevel)
    {
        if (requiredEqualityLevel == null) {
            return mRuleManager.compare(child1, child2);
        }
        else if (mRuleManager.areAtLeastEqual(child1, child2, requiredEqualityLevel)) {
            return requiredEqualityLevel;
        }
        else {
            return NodeEquality.UNEQUAL;
        }
    }
}
