//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.List;

import org.junit.Test;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.IEventTarget;
import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.eventcore.StringEventType;
import de.ugoe.cs.autoquest.eventcore.gui.KeyPressed;
import de.ugoe.cs.autoquest.eventcore.gui.KeyReleased;
import de.ugoe.cs.autoquest.eventcore.gui.KeyTyped;
import de.ugoe.cs.autoquest.eventcore.gui.KeyboardFocusChange;
import de.ugoe.cs.autoquest.eventcore.gui.MouseButtonDown;
import de.ugoe.cs.autoquest.eventcore.gui.MouseButtonInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.MouseButtonUp;
import de.ugoe.cs.autoquest.eventcore.gui.MouseClick;
import de.ugoe.cs.autoquest.eventcore.gui.MouseDoubleClick;
import de.ugoe.cs.autoquest.eventcore.gui.MouseDragAndDrop;
import de.ugoe.cs.autoquest.eventcore.gui.Scroll;
import de.ugoe.cs.autoquest.eventcore.gui.TextInput;
import de.ugoe.cs.autoquest.eventcore.gui.TextSelection;
import de.ugoe.cs.autoquest.eventcore.gui.ValueSelection;
import de.ugoe.cs.autoquest.keyboardmaps.VirtualKey;
import de.ugoe.cs.autoquest.tasktrees.taskequality.EventTaskComparisonRule;
import de.ugoe.cs.autoquest.tasktrees.taskequality.GUIEventTaskComparisonRule;
import de.ugoe.cs.autoquest.tasktrees.taskequality.TaskEquality;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskFactory;
import de.ugoe.cs.autoquest.tasktrees.treeimpl.TaskFactory;
import de.ugoe.cs.autoquest.test.DummyGUIElement;

/**
 * @author Patrick Harms
 */
public class GUIEventTaskComparisonRuleTest {

    /**
     *
     */
    @Test
    public void test_isApplicable_01() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        assertTrue(rule.isApplicable(task1, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_02() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new StringEventType("eventType2");
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_03() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new KeyboardFocusChange();
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new KeyboardFocusChange();
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_04() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new KeyPressed(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new KeyReleased(VirtualKey.LETTER_B);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_05() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new KeyTyped(VirtualKey.LETTER_C);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new KeyTyped(VirtualKey.LETTER_D);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_06() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 5, 6);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new MouseButtonUp(MouseButtonInteraction.Button.MIDDLE, 3, 1);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_07() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new MouseClick(MouseButtonInteraction.Button.RIGHT, 4, 7);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new MouseDoubleClick(MouseButtonInteraction.Button.X, 9, 12);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_08() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new MouseDragAndDrop(1, 2, 3, 4);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new MouseDragAndDrop(5, 6, 7, 8);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_09() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new Scroll(1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new Scroll(3, 4);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_10() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        List<Event> inputEvents = new ArrayList<Event>();
        IEventType eventType1 = new TextInput("text1", inputEvents);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new TextInput("text2", inputEvents);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_11() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new TextSelection();
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new ValueSelection<String>("value");
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_12() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        ITask selection = taskFactory.createNewSelection();

        assertFalse(rule.isApplicable(task1, selection));
        assertFalse(rule.isApplicable(selection, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_13() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        ITask sequence = taskFactory.createNewSequence();

        assertFalse(rule.isApplicable(task1, sequence));
        assertFalse(rule.isApplicable(sequence, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_14() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        ITask iteration = taskFactory.createNewIteration();

        assertFalse(rule.isApplicable(task1, iteration));
        assertFalse(rule.isApplicable(iteration, task1));
    }

    /**
     *
     */
    @Test
    public void test_isApplicable_15() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        ITask optional = taskFactory.createNewOptional();

        assertFalse(rule.isApplicable(task1, optional));
        assertFalse(rule.isApplicable(optional, task1));
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyboardFocusChange_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyboardFocusChange();
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyboardFocusChange_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyboardFocusChange();
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new KeyboardFocusChange();
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyboardFocusChange_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyboardFocusChange();
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new KeyboardFocusChange();
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyPressed_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyPressed(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyPressed_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyPressed(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new KeyPressed(VirtualKey.LETTER_A);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyPressed_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyPressed(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new KeyPressed(VirtualKey.LETTER_B);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyPressed_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyPressed(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new KeyPressed(VirtualKey.LETTER_A);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_KeyReleased_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyReleased(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyReleased_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyReleased(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new KeyReleased(VirtualKey.LETTER_A);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyReleased_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyReleased(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new KeyReleased(VirtualKey.LETTER_B);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyReleased_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyReleased(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new KeyReleased(VirtualKey.LETTER_A);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_KeyTyped_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyTyped(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyTyped_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyTyped(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new KeyTyped(VirtualKey.LETTER_A);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyTyped_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyTyped(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new KeyTyped(VirtualKey.LETTER_B);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_KeyTyped_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new KeyTyped(VirtualKey.LETTER_A);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new KeyTyped(VirtualKey.LETTER_A);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_MouseButtonDown_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseButtonDown_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 1, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseButtonDown_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 1, 3);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseButtonDown_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 3, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseButtonDown_05() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonDown(MouseButtonInteraction.Button.RIGHT, 1, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_MouseButtonDown_06() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonDown(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_MouseButtonUp_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseButtonUp_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 1, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseButtonUp_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 1, 3);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseButtonUp_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 3, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseButtonUp_05() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonUp(MouseButtonInteraction.Button.RIGHT, 1, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_MouseButtonUp_06() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseButtonUp(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_MouseClick_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseClick_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseClick_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseClick(MouseButtonInteraction.Button.LEFT, 1, 3);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseClick_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseClick(MouseButtonInteraction.Button.LEFT, 3, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseClick_05() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseClick(MouseButtonInteraction.Button.RIGHT, 1, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_MouseClick_06() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_MouseDoubleClick_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDoubleClick_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDoubleClick_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 1, 3);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDoubleClick_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 3, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDoubleClick_05() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDoubleClick(MouseButtonInteraction.Button.RIGHT, 1, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_MouseDoubleClick_06() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDoubleClick(MouseButtonInteraction.Button.LEFT, 1, 2);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_MouseDragAndDrop_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDragAndDrop(1, 2, 3, 4);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDragAndDrop_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDragAndDrop(1, 2, 3, 4);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDragAndDrop(1, 2, 3, 4);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDragAndDrop_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDragAndDrop(1, 2, 3, 4);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDragAndDrop(5, 2, 3, 4);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDragAndDrop_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDragAndDrop(1, 2, 3, 4);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDragAndDrop(1, 5, 3, 4);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDragAndDrop_05() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDragAndDrop(1, 2, 3, 4);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDragAndDrop(1, 2, 5, 4);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDragAndDrop_06() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDragAndDrop(1, 2, 3, 4);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDragAndDrop(1, 2, 3, 5);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_MouseDragAndDrop_07() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new MouseDragAndDrop(1, 2, 3, 4);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new MouseDragAndDrop(1, 2, 3, 4);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_Scroll_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new Scroll(1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_Scroll_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new Scroll(1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new Scroll(1, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_Scroll_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new Scroll(1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new Scroll(3, 2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_Scroll_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new Scroll(1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new Scroll(1, 3);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_Scroll_05() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new Scroll(1, 2);
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new Scroll(1, 2);
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_TextInput_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        List<Event> textInputEvents1 = new ArrayList<Event>();
        textInputEvents1.add(new Event(eventType1, eventTarget1));
        eventType1 = new TextInput("enteredText1", textInputEvents1);
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_TextInput_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        List<Event> textInputEvents1 = new ArrayList<Event>();
        textInputEvents1.add(new Event(eventType1, eventTarget1));
        eventType1 = new TextInput("enteredText1", textInputEvents1);
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new TextInput("enteredText1", textInputEvents1);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);

        assertLexicallyEqual(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_TextInput_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        List<Event> textInputEvents1 = new ArrayList<Event>();
        textInputEvents1.add(new Event(eventType1, eventTarget1));
        eventType1 = new TextInput("enteredText1", textInputEvents1);
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new StringEventType("eventType2");
        List<Event> textInputEvents2 = new ArrayList<Event>();
        textInputEvents2.add(new Event(eventType2, eventTarget1));
        eventType2 = new TextInput("enteredText1", textInputEvents2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);

        assertSyntacticallyEqual(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_TextInput_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        List<Event> textInputEvents1 = new ArrayList<Event>();
        textInputEvents1.add(new Event(eventType1, eventTarget1));
        eventType1 = new TextInput("enteredText1", textInputEvents1);
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new StringEventType("eventType2");
        List<Event> textInputEvents2 = new ArrayList<Event>();
        textInputEvents2.add(new Event(eventType2, eventTarget1));
        eventType2 = new TextInput("enteredText2", textInputEvents2);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);

        assertSemanticallyEqual(task1, task2);
    }
    
    /**
     *
     */
    @Test
    public void test_compare_TextInput_05() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        List<Event> textInputEvents1 = new ArrayList<Event>();
        textInputEvents1.add(new Event(eventType1, eventTarget1));
        eventType1 = new TextInput("enteredText1", textInputEvents1);
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType1, eventTarget2);

        assertUnequal(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_TextSelection_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new TextSelection();
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_TextSelection_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new TextSelection();
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new TextSelection();
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_TextSelection_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new TextSelection();
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new TextSelection();
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_ValueSelection_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new ValueSelection<String>("value1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertLexicallyEqual(task1, task1);
    }

    /**
     *
     */
    @Test
    public void test_compare_ValueSelection_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new ValueSelection<String>("value1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new ValueSelection<String>("value1");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertLexicallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_ValueSelection_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new ValueSelection<String>("value1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new ValueSelection<String>("value2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);
        
        assertSemanticallyEqual(task1, task2);
    }

    /**
     *
     */
    @Test
    public void test_compare_ValueSelection_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        IEventType eventType1 = new ValueSelection<String>("value1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new ValueSelection<String>("value1");
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertUnequal(task1, task2);
    }
    
    /**
     *
     */
    private void assertLexicallyEqual(ITask task1, ITask task2) {
        GUIEventTaskComparisonRule rule = new GUIEventTaskComparisonRule();
        
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(task1, task2));
        assertTrue(rule.areLexicallyEqual(task1, task2));
        assertTrue(rule.areSyntacticallyEqual(task1, task2));
        assertTrue(rule.areSemanticallyEqual(task1, task2));

        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(task2, task1));
        assertTrue(rule.areLexicallyEqual(task2, task1));
        assertTrue(rule.areSyntacticallyEqual(task2, task1));
        assertTrue(rule.areSemanticallyEqual(task2, task1));
    }

    /**
     *
     */
    private void assertSyntacticallyEqual(ITask task1, ITask task2) {
        GUIEventTaskComparisonRule rule = new GUIEventTaskComparisonRule();
        
        assertEquals(TaskEquality.SYNTACTICALLY_EQUAL, rule.compare(task1, task2));
        assertFalse(rule.areLexicallyEqual(task1, task2));
        assertTrue(rule.areSyntacticallyEqual(task1, task2));
        assertTrue(rule.areSemanticallyEqual(task1, task2));

        assertEquals(TaskEquality.SYNTACTICALLY_EQUAL, rule.compare(task2, task1));
        assertFalse(rule.areLexicallyEqual(task2, task1));
        assertTrue(rule.areSyntacticallyEqual(task2, task1));
        assertTrue(rule.areSemanticallyEqual(task2, task1));
    }

    /**
     *
     */
    private void assertSemanticallyEqual(ITask task1, ITask task2) {
        GUIEventTaskComparisonRule rule = new GUIEventTaskComparisonRule();
        
        assertEquals(TaskEquality.SEMANTICALLY_EQUAL, rule.compare(task1, task2));
        assertFalse(rule.areLexicallyEqual(task1, task2));
        assertFalse(rule.areSyntacticallyEqual(task1, task2));
        assertTrue(rule.areSemanticallyEqual(task1, task2));

        assertEquals(TaskEquality.SEMANTICALLY_EQUAL, rule.compare(task2, task1));
        assertFalse(rule.areLexicallyEqual(task2, task1));
        assertFalse(rule.areSyntacticallyEqual(task2, task1));
        assertTrue(rule.areSemanticallyEqual(task2, task1));
    }

    /**
     *
     */
    private void assertUnequal(ITask task1, ITask task2) {
        GUIEventTaskComparisonRule rule = new GUIEventTaskComparisonRule();
        
        assertEquals(TaskEquality.UNEQUAL, rule.compare(task1, task2));
        assertFalse(rule.areLexicallyEqual(task1, task2));
        assertFalse(rule.areSyntacticallyEqual(task1, task2));
        assertFalse(rule.areSemanticallyEqual(task1, task2));

        assertEquals(TaskEquality.UNEQUAL, rule.compare(task2, task1));
        assertFalse(rule.areLexicallyEqual(task2, task1));
        assertFalse(rule.areSyntacticallyEqual(task2, task1));
        assertFalse(rule.areSemanticallyEqual(task2, task1));
    }
}
