//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.ui.swt;

import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Dialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IMarkingTemporalRelationship;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IStructuringTemporalRelationship;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstanceList;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IUserSession;

import org.eclipse.swt.widgets.Label;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Patrick Harms
 */
public class ShowTaskTreeDialog extends Dialog {

    /** */
    protected Shell shell;
    
    /** */
    private Tree instanceTree;
    
    /** */
    private Tree modelTree;

    /** */
    protected ITaskModel taskModel;

    /**
     * 
     */
    public ShowTaskTreeDialog(Shell parent, int style, ITaskModel taskModel, String taskTreeName) {
        super(parent, style);
        setText("Task Model " + taskTreeName);
        this.taskModel = taskModel;
    }

    /**
     * 
     */
    public void open() {
        createContents();
        shell.open();
        shell.layout();
        Display display = getParent().getDisplay();
        while (!shell.isDisposed()) {
            if (!display.readAndDispatch()) {
                display.sleep();
            }
        }
    }

    /**
     * 
     */
    private void createContents() {
        shell = new Shell(getParent(), SWT.SHELL_TRIM | SWT.BORDER | SWT.APPLICATION_MODAL);
        shell.setSize(450, 300);
        shell.setText(getText());

        shell.setLayout(new GridLayout(4, false));
        
        SashForm sashForm = new SashForm(shell, SWT.VERTICAL);
        sashForm.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 4, 1));
        
        instanceTree = new Tree(sashForm, SWT.BORDER | SWT.MULTI);
        instanceTree.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                modelTree.removeAll();
                TreeItem[] selectedItems = instanceTree.getSelection();
                if ((selectedItems.length == 1) &&
                    (selectedItems[0].getData() instanceof ITaskInstance))
                {
                    buildModelTree((ITaskInstance) selectedItems[0].getData());
                }
            }
        });

        buildInstanceTree();

        modelTree = new Tree(sashForm, SWT.BORDER | SWT.MULTI);

        Button btnExpandAll = new Button(shell, SWT.NONE);
        btnExpandAll.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                expandAll(instanceTree, true);
            }
        });
        btnExpandAll.setText("Expand all");

        Button btnCollapseAll = new Button(shell, SWT.NONE);
        btnCollapseAll.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                expandAll(instanceTree, false);
            }
        });
        btnCollapseAll.setText("Collapse all");
        
        //new Label(shell, SWT.NONE);
        new Label(shell, SWT.NONE);
        new Label(shell, SWT.NONE);
        new Label(shell, SWT.NONE);

    }

    /**
     * 
     */
    private void buildInstanceTree() {
        List<IUserSession> sessions = taskModel.getUserSessions();
        
        TreeItem root = new TreeItem(instanceTree, SWT.NULL);
        root.setText(sessions.size() + " sessions");
        root.setData(sessions);
        
        for (IUserSession session : sessions) {
            buildInstanceTree(root, session);
        }
    }

    /**
     * 
     */
    private void buildInstanceTree(TreeItem currentParent, ITaskInstanceList taskInstanceList) {
        TreeItem child = new TreeItem(currentParent, SWT.NULL);
        child.setText(taskInstanceList.toString());
        child.setData(taskInstanceList);
        
        for (ITaskInstanceList childTask : taskInstanceList) {
            buildInstanceTree(child, childTask);
        }
    }

    /**
     * 
     */
    private void buildModelTree(ITaskInstance taskInstance) {
        modelTree.removeAll();
        TreeItem root = new TreeItem(modelTree, SWT.NULL);
        root.setText("model of instance " + taskInstance);
        root.setData(taskInstance);

        buildModelTree(taskInstance.getTask(), root);
        expandAll(root, true);
    }

    /**
     * 
     */
    private void buildModelTree(ITask task, TreeItem parent) {
        TreeItem node = new TreeItem(parent, SWT.NULL);
        node.setText(task.toString());
        node.setData(task);
        
        if (task instanceof IStructuringTemporalRelationship) {
            for (ITask subTask : ((IStructuringTemporalRelationship) task).getChildren()) {
                buildModelTree(subTask, node);
            }
        }
        else if (task instanceof IMarkingTemporalRelationship) {
            buildModelTree(((IMarkingTemporalRelationship) task).getMarkedTask(), node);
        }
    }

    /**
     * 
     */
    private void expandAll(Tree tree, boolean expanded) {
        for (TreeItem item : tree.getItems()) {
            expandAll(item, expanded);
        }
    }

    /**
     * 
     */
    private void expandAll(TreeItem item, boolean expanded) {
        item.setExpanded(expanded);
        for (TreeItem childItem : item.getItems()) {
            expandAll(childItem, expanded);
        }
    }
    
}
