//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import static org.junit.Assert.*;

import org.junit.Test;

import de.ugoe.cs.autoquest.eventcore.IEventTarget;
import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.eventcore.StringEventType;
import de.ugoe.cs.autoquest.tasktrees.taskequality.IterationComparisonRule;
import de.ugoe.cs.autoquest.tasktrees.taskequality.TaskEquality;
import de.ugoe.cs.autoquest.tasktrees.taskequality.TaskEqualityRuleManager;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskBuilder;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskFactory;
import de.ugoe.cs.autoquest.tasktrees.treeimpl.TaskBuilder;
import de.ugoe.cs.autoquest.tasktrees.treeimpl.TaskFactory;
import de.ugoe.cs.autoquest.test.DummyGUIElement;

/**
 * @author Patrick Harms
 */
public class IterationComparisonRuleTest {

    /**
     *
     */
    @Test
    public void test() {
        TaskEqualityRuleManager manager = new TaskEqualityRuleManager();
        manager.init();
        
        ITaskFactory taskFactory = new TaskFactory();
        ITaskBuilder treeBuilder = new TaskBuilder();
        
        IterationComparisonRule rule = new IterationComparisonRule(manager);
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");

        IEventType eventType2 = new StringEventType("eventType2");
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");

        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertFalse(rule.isApplicable(task1, task2));
        
        IIteration iteration1 = taskFactory.createNewIteration();
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration1, iteration1));

        IIteration iteration2 = taskFactory.createNewIteration();
        
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration1, iteration2));
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration2, iteration1));
        
        treeBuilder.setMarkedTask(iteration1, task1);
        
        assertEquals(TaskEquality.UNEQUAL, rule.compare(iteration1, iteration2));
        assertEquals(TaskEquality.UNEQUAL, rule.compare(iteration2, iteration1));
        
        treeBuilder.setMarkedTask(iteration2, task1);
        
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration1, iteration2));
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration2, iteration1));
        
        treeBuilder.setMarkedTask(iteration1, task2);
        
        assertEquals(TaskEquality.UNEQUAL, rule.compare(iteration1, iteration2));
        assertEquals(TaskEquality.UNEQUAL, rule.compare(iteration2, iteration1));
        
        treeBuilder.setMarkedTask(iteration2, task2);
        
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration1, iteration2));
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration2, iteration1));
        
        ISelection selection1 = taskFactory.createNewSelection();
        treeBuilder.addChild(selection1, task2);
        treeBuilder.setMarkedTask(iteration1, selection1);
        
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration1, iteration2));
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration2, iteration1));
        
        ISelection selection2 = taskFactory.createNewSelection();
        treeBuilder.addChild(selection2, task2);
        treeBuilder.setMarkedTask(iteration2, selection2);
        
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration1, iteration2));
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(iteration2, iteration1));
        
        assertFalse(rule.isApplicable(iteration1, selection1));
        assertFalse(rule.isApplicable(selection1, iteration1));
        assertFalse(rule.isApplicable(iteration2, selection1));
        assertFalse(rule.isApplicable(selection1, iteration2));

        assertFalse(rule.isApplicable(iteration1, selection2));
        assertFalse(rule.isApplicable(selection2, iteration1));
        assertFalse(rule.isApplicable(iteration2, selection2));
        assertFalse(rule.isApplicable(selection2, iteration2));
    }

}
