//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;

/**
 * <p>
 * This class is capable of comparing any task which is not an iteration with an
 * iteration. This is needed, because iterations may iterate exactly that task. In this
 * case, the iteration would be equal to that task if it was executed exactly once. The rule
 * returns lexically equal, it the child of the iteration is lexically equal to the task
 * or if the child of the iteration is a selection and this selections contains a lexically equal
 * task. The same applies for syntactical and semantical equality.
 * </p>

 * @author Patrick Harms
 */
public class TaskAndIterationComparisonRule implements TaskComparisonRule {
    
    /** the rule manager for internally comparing tasks */
    private TaskEqualityRuleManager mRuleManager;

    /**
     * <p>
     * simple constructor to provide the rule with the task equality rule manager to be able
     * to perform comparisons of the children of provided tasks
     * </p>
     * 
     * @param ruleManager the rule manager for comparing tasks
     */
    TaskAndIterationComparisonRule(TaskEqualityRuleManager ruleManager) {
        super();
        mRuleManager = ruleManager;
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#isApplicable(ITask, ITask)
     */
    @Override
    public boolean isApplicable(ITask task1, ITask task2) {
        return ((task1 instanceof IIteration) && (!(task2 instanceof IIteration))) ||
               ((task2 instanceof IIteration) && (!(task1 instanceof IIteration)));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areLexicallyEqual(ITask, ITask)
     */
    @Override
    public boolean areLexicallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.LEXICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSyntacticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.SYNTACTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSemanticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSemanticallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.SEMANTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#compare(ITask, ITask)
     */
    @Override
    public TaskEquality compare(ITask task1, ITask task2) {
        return getEquality(task1, task2, null);
    }

    /**
     * 
     */
    private TaskEquality getEquality(ITask task1, ITask task2, TaskEquality requiredEqualityLevel) {
        IIteration iteration = null;
        ITask task = null;
        
        if (task1 instanceof IIteration) {
            if (task2 instanceof IIteration) {
                // the rule is not responsible for two iterations
                return null;
            }
            
            iteration = (IIteration) task1;
            task = task2;
        }
        else if (task2 instanceof IIteration) {
            if (task1 instanceof IIteration) {
                // the rule is not responsible for two iterations
                return null;
            }
            
            iteration = (IIteration) task2;
            task = task1;
        }
        else {
            return null;
        }

        ITask child = iteration.getMarkedTask();
        
        // now, that we found the iteration and the task, lets compare the child of the iteration
        // with the task.
        if (child == null) {
            return null;
        }

        TaskEquality taskEquality = callRuleManager(child, task, requiredEqualityLevel);

        // although the subtask may be identical to the task, we can not return identical, as
        // the iteration is not identical to the task, but at most lexically equal
        if (taskEquality == TaskEquality.IDENTICAL) {
            return TaskEquality.LEXICALLY_EQUAL;
        }
        else {
            return taskEquality;
        }

    }
    
    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param child1
     * @param child2
     * @param requiredEqualityLevel
     * @return
     */
    private TaskEquality callRuleManager(ITask        child1,
                                         ITask        child2,
                                         TaskEquality requiredEqualityLevel)
    {
        if (requiredEqualityLevel == null) {
            return mRuleManager.compare(child1, child2);
        }
        else if (mRuleManager.areAtLeastEqual(child1, child2, requiredEqualityLevel)) {
            return requiredEqualityLevel;
        }
        else {
            return TaskEquality.UNEQUAL;
        }
    }
}
