//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules.metrics;

import static de.ugoe.cs.autoquest.usability.tasktree.filters.EventTypeFilter.TEXT_INPUT;

import java.util.List;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTree;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;
import de.ugoe.cs.autoquest.usability.EvaluationMethodCaller;
import de.ugoe.cs.autoquest.usability.result.DefectDescriptionResolver;
import de.ugoe.cs.autoquest.usability.result.UsabilityDefect;
import de.ugoe.cs.autoquest.usability.rules.UsabilityMetric;
import de.ugoe.cs.autoquest.usability.rules.UsabilityRule;
import de.ugoe.cs.autoquest.usability.tasktree.FilterResult;
import de.ugoe.cs.autoquest.usability.tasktree.IterativeDFSFilterStrategy;
import de.ugoe.cs.autoquest.usability.tasktree.filters.TaskTreeFilter;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Alexander Deicke
 */
public class TextInputRatioMetric extends UsabilityRule implements UsabilityMetric {

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskTree
     */
    public TextInputRatioMetric(ITaskTree taskTree) {
        super(taskTree);
        this.name = "TextInputRatio";
        this.defect = new DefectDescriptionResolver().descriptionFor(this.getClass().getSimpleName());
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usability.rules.UsabilityRule#check()
     */
    @Override
    public Optional<UsabilityDefect> calculate() {
        FilterResult textInputEvents = extractNodesFromTaskTree();
        float evaluationMetric = calculateEvaluationMetric(textInputEvents.nodesMatchedFilter(), textInputEvents.nodesNotMatchedFilter());
        return this.defect.isPresent(evaluationMetric);
    }
    
    private FilterResult extractNodesFromTaskTree() {
        return new TaskTreeFilter(new IterativeDFSFilterStrategy()).filterByEventType(TEXT_INPUT)
            .from(this.taskTree);
    }
    
    private float calculateEvaluationMetric(List<ITaskTreeNode> textInputEvents, List<ITaskTreeNode> nonTextInputEvents) {
        float numberOfTextInputEvents = textInputEvents.size();
        float numberOfNonTextInputEvents = nrOfEventNodesNotMatchedFilter(nonTextInputEvents);
        return numberOfTextInputEvents / (numberOfTextInputEvents + numberOfNonTextInputEvents);
    }

    private int nrOfEventNodesNotMatchedFilter(List<ITaskTreeNode> nonTextInputEvents) {
        return Iterables.size(Iterables.filter(nonTextInputEvents,
                                               new Predicate<ITaskTreeNode>() {

                                                   @Override
                                                   public boolean apply(ITaskTreeNode node) {
                                                       return (node.getChildren() == null) ||
                                                           (node.getChildren().size() == 0);
                                                   }
                                               }));
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usability.rules.UsabilityRule#callEvaluationMetho(de.ugoe.cs.autoquest.usability.EvaluationMethodCaller)
     */
    @Override
    public Optional<UsabilityDefect> callEvaluationMethod(EvaluationMethodCaller evaluationMethodCaller)
    {
        return evaluationMethodCaller.check(this);
    }
}
