//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules.patterns;

import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.google.common.base.Splitter;
import com.google.common.collect.Lists;

import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.eventcore.StringEventType;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTree;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTreeNode;
import de.ugoe.cs.autoquest.tasktrees.treeifc.NodeVisitor;
import de.ugoe.cs.autoquest.usability.tasktree.filters.EventTypeFilter;
import de.ugoe.cs.autoquest.usability.util.PatternsVisitorUtil;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Alexander Deicke
 */
public abstract class UsagePatternVisitor implements NodeVisitor {
    
    protected EventTypeFilter eventType;
    
    protected UsagePattern containedPattern;
    
    protected boolean present = false;
    
    protected List<ITaskTreeNode> retainedChildrenNodesFromSelectionNodes = Lists.newArrayList();
    
    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.NodeVisitor#visit(de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask)
     */
    public void visit(IEventTask event) {
        if(!this.present && isEventVisitor()) {
            IEventType eventType = event.getEventType();
            if(eventType instanceof StringEventType) {
                this.present = eventType.toString().equals(nameOfEventType());
            } else {
                this.present = eventType.getClass().equals(this.eventType.clazz());
            }
        }
    }
    
    public boolean isEventVisitor() {
        return this.eventType != null && this.containedPattern == null;
    }
    
    protected String nameOfEventType() {
        String ret = StringUtils.EMPTY;
        Iterable<String> splitted = Splitter.on("_").split(this.eventType.name());
        for(String str : splitted) {
            str = str.toLowerCase();
            ret += Character.toString(str.charAt(0)).toUpperCase() + str.substring(1);
        }
        return ret;
    }
    
    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.treeifc.NodeVisitor#visit(de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection)
     */
    public void visit(ISelection selection) {
        if(isEventVisitor()) {
            retainNodesWherePatternIsPresent(selection.getChildren());
            this.present = patternIsPresent();
        } else {
            ITaskTree taskTree = PatternsVisitorUtil.createTaskTreeFromNode(selection);
            this.present = containedPattern.containedIn(taskTree);  
        }
    }

    protected void retainNodesWherePatternIsPresent(List<ITaskTreeNode> children) {
        for(ITaskTreeNode node : children) {
            this.present = false;
            node.accept(this);
            if(this.present) {
                this.retainedChildrenNodesFromSelectionNodes.add(node);
            }
        }
    }
    
    private boolean patternIsPresent() {
        return !this.retainedChildrenNodesFromSelectionNodes.isEmpty();
    }
    
    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @return
     */
    public boolean isPresent() {
        return this.present;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     */
    public void reset() {
        this.retainedChildrenNodesFromSelectionNodes.clear();
        this.present = false;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @return
     */
    public boolean hasExcludedSelectionNodes() {
        return patternIsPresent();
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @return
     */
    public List<ITaskTreeNode> getRetainedSelectionNodes() {
        return this.retainedChildrenNodesFromSelectionNodes;
    }
    
}
