//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;

/**
 * <p>
 * This rule is capable of comparing sequences. If both sequences do not have children, they are
 * treated as lexically equal. Sequences are lexically equal, if they have the same number and
 * order of lexically equal children. The rule can not decide, if two sequences are syntactically
 * or semantically equal.
 * </p>
 * 
 * @version $Revision: $ $Date: 19.02.2012$
 * @author 2012, last modified by $Author: patrick$
 */
public class SequenceComparisonRule implements TaskComparisonRule {

    /** the rule manager for internally comparing tasks */
    private TaskEqualityRuleManager mRuleManager;

    /**
     * <p>
     * simple constructor to provide the rule with the task equality rule manager to be able
     * to perform comparisons of the children of provided tasks
     * </p>
     * 
     * @param ruleManager the rule manager for comparing tasks
     */
    SequenceComparisonRule(TaskEqualityRuleManager ruleManager) {
        super();
        mRuleManager = ruleManager;
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#isApplicable(ITask, ITask)
     */
    @Override
    public boolean isApplicable(ITask task1, ITask task2) {
        return (task1 instanceof ISequence) && (task2 instanceof ISequence);
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areLexicallyEqual(ITask, ITask)
     */
    @Override
    public boolean areLexicallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.LEXICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSyntacticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.SYNTACTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSemanticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSemanticallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.SEMANTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#compare(ITask, ITask)
     */
    @Override
    public TaskEquality compare(ITask task1, ITask task2) {
        return getEquality(task1, task2, null);
    }

    /**
     * 
     */
    private TaskEquality getEquality(ITask task1, ITask task2, TaskEquality requiredEqualityLevel) {
        List<ITask> children1 = ((ISequence) task1).getChildren();
        List<ITask> children2 = ((ISequence) task2).getChildren();

        // if both sequences do not have children, they are equal although this doesn't make sense
        if ((children1.size() == 0) && (children2.size() == 0)) {
            return TaskEquality.LEXICALLY_EQUAL;
        }

        if (children1.size() != children2.size()) {
            return TaskEquality.UNEQUAL;
        }

        TaskEquality resultingEquality = TaskEquality.LEXICALLY_EQUAL;
        for (int i = 0; i < children1.size(); i++) {
            ITask child1 = children1.get(i);
            ITask child2 = children2.get(i);

            TaskEquality taskEquality = callRuleManager(child1, child2, requiredEqualityLevel);

            if ((taskEquality == null) || (taskEquality == TaskEquality.UNEQUAL)) {
                return TaskEquality.UNEQUAL;
            }
            
            resultingEquality = resultingEquality.getCommonDenominator(taskEquality);
        }

        return resultingEquality;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param child1
     * @param child2
     * @param requiredEqualityLevel
     * @return
     */
    private TaskEquality callRuleManager(ITask        child1,
                                         ITask        child2,
                                         TaskEquality requiredEqualityLevel)
    {
        if (requiredEqualityLevel == null) {
            return mRuleManager.compare(child1, child2);
        }
        else if (mRuleManager.areAtLeastEqual(child1, child2, requiredEqualityLevel)) {
            return requiredEqualityLevel;
        }
        else {
            return TaskEquality.UNEQUAL;
        }
    }
}
