//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules.metrics;

import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.absent;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.highRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.infoRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.lowRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.mediumRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.present;
import static org.fest.assertions.api.Assertions.assertThat;

import org.junit.Test;

import com.google.common.base.Optional;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskTree;
import de.ugoe.cs.autoquest.usability.result.UsabilityDefect;
import de.ugoe.cs.autoquest.usability.rules.metrics.TextInputRatioMetric;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Alexander Deicke
 */
public class TextInputRatioEvaluatorTest extends AbstractUsabilityEvaluationTC {

    @Test
    public void should_return_no_recommendation() {
        // Given
        String spec =
            "Sequence {" + "  Interaction {}" + "  TextInput (a) {}" + "  Interaction {}"
                + "  Interaction {}" + "  TextInput (c) {}" + "  Interaction {}"
                + "  Interaction {}" + "  Interaction {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(absent());
    }

    @Test
    public void should_return_recommendation_with_info_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  Interaction {}" + "  TextInput (a) {}" + "  Interaction {}"
                + "  Interaction {}" + "  TextInput (c) {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(infoRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_low_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  Interaction {}" + "  TextInput (a) {}" + "  TextInput (b) {}"
                + "  Interaction {}" + "  TextInput (c) {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(lowRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_medium_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a) {}" + "  TextInput (b) {}" + "  Interaction {}"
                + "  TextInput (c) {}" + "}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(mediumRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_high_severity_level() {
        // Given
        String spec = "TextInput (bla) {}";
        ITaskTree taskTree = createTaskTree(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(highRecommendationSeverityLevel());
    }
}
