//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.treeifc;

import java.io.Serializable;

/**
 * <p>
 * A task represents a model for events that occur if the user interacts with a software for
 * achieving a specific goal. A task can be a single event or a complex structure of events
 * and temporal relationships defining the event order. Tasks may especially refer to other tasks
 * to create task structures similar to trees. These structures fully define in which ways the
 * events that form the task can occur.
 * </p>
 * 
 * @author Patrick Harms
 */
public interface ITask extends Cloneable, Serializable {

    /**
     * <p>
     * every task is assigned a unique id which is returned by this method. The id is unique for
     * the current runtime.
     * </p>
     * 
     * @return as described
     */
    public int getId();

    /**
     * <p>
     * returns a human readable description for task. 
     * </p>
     * 
     * @return as described
     */
    public String getDescription();

    /**
     * <p>
     * checks whether this task is equal to another one. Task equality is only given, if two
     * tasks have the same id. This means, that this method must only return true if the other
     * task is either the same object or a clone of it. 
     * </p>
     * 
     * @return as described
     */
    public boolean equals(ITask task);

    /**
     * <p>
     * returns a hash code for the task, which is usually the id returned by {@link #getId()}.
     * </p>
     * 
     * @return as described
     */
    public int hashCode();

    /**
     * <p>
     * returns an exact copy of this task. The clone has the same id. If the task has
     * children, they are cloned as well. A call on the method {@link #equals(ITask)} with the
     * result of this method must return true.
     * </p>
     * 
     * @return as described
     */
    public ITask clone();
    
    /**
     * <p>
     * implements the visitor pattern to be able to process tasks and their children.
     * </p>
     * 
     * @param visitor the visitor used to process the task
     */
    public void accept(ITaskVisitor visitor);
    
}
