//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;

/**
 * <p>
 * This class is capable of comparing Iterations. Iterations equal at distinct levels
 * in distinct situations. The following table shows the results of the comparison for the
 * specific situations (the parameters are commutative). In any other situation, the comparison
 * returns <code>NodeEquality.UNEQUAL</code>:
 * </p>
 * 
 * <table border="1">
 *   <tr>
 *     <th>iteration 1</th>
 *     <th>iteration 2</th>
 *     <th>comparison result</th>
 *   </tr>
 *   <tr>
 *     <td>any iteration</td>
 *     <td>any iteration with a child that is lexically equal to the child of iteration 1</td>
 *     <td><code>NodeEquality.LEXICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>any iteration</td>
 *     <td>any iteration with a child that is syntactically equal to the child of iteration 1</td>
 *     <td><code>NodeEquality.SYNTACTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>any iteration</td>
 *     <td>any iteration with a child that is semantically equal to the child of iteration 1</td>
 *     <td><code>NodeEquality.SEMANTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>an iteration with a selection of syntactically equal children</td>
 *     <td>an iteration with a child that is syntactically equal to the children of the child
 *     selection of iteration 1</td>
 *     <td><code>NodeEquality.SYNTACTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>an iteration with a selection of syntactically equal children</td>
 *     <td>an iteration with a selection of syntactically equal children that are all syntactically
 *     equal to the selection of children of iteration 1</td>
 *     <td><code>NodeEquality.SYNTACTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>an iteration with a selection of semantically equal children</td>
 *     <td>an iteration with a child that is semantically equal to the children of the child
 *     selection of iteration 1</td>
 *     <td><code>NodeEquality.SEMANTICALLY_EQUAL</code></td>
 *   </tr>
 *   <tr>
 *     <td>an iteration with a selection of semantically equal children</td>
 *     <td>an iteration with a selection of semantically equal children that are all semantically
 *     equal to the selection of children of iteration 1</td>
 *     <td><code>NodeEquality.SEMANTICALLY_EQUAL</code></td>
 *   </tr>
 * </table>
 * 
 * @version $Revision: $ $Date: 19.02.2012$
 * @author 2012, last modified by $Author: patrick$
 */
public class IterationComparisonRule implements TaskComparisonRule {
    
    /** the rule manager for internally comparing tasks */
    private TaskEqualityRuleManager mRuleManager;

    /**
     * <p>
     * simple constructor to provide the rule with the task equality rule manager to be able
     * to perform comparisons of the children of provided tasks
     * </p>
     * 
     * @param ruleManager the rule manager for comparing tasks
     */
    IterationComparisonRule(TaskEqualityRuleManager ruleManager) {
        super();
        mRuleManager = ruleManager;
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#isApplicable(ITask, ITask)
     */
    @Override
    public boolean isApplicable(ITask task1, ITask task2) {
        return (task1 instanceof IIteration) && (task2 instanceof IIteration);
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areLexicallyEqual(ITask, ITask)
     */
    @Override
    public boolean areLexicallyEqual(ITask task1, ITask task2) {
        ITask child1 = ((IIteration) task1).getMarkedTask();
        ITask child2 = ((IIteration) task2).getMarkedTask();
        
        if (child1 != null) {
            if (child2 == null) {
                return false;
            }
            else {
                // iterations may have 3 different structures.
                // 1. they have one child, which is the iterated one
                // 2. they have a sequence of children, which is iterated
                // 3. they have a selection of different iterated variants (usually the variants
                //    are semantically equal)
                // check if the type of children match. If not, return false. If they match,
                // use the equality manager to perform further comparisons
                
                if (((child1 instanceof ISelection) && (child2 instanceof ISelection)) ||
                    ((child1 instanceof ISequence) && (child2 instanceof ISequence)) ||
                    ((child1 instanceof IEventTask) && (child2 instanceof IEventTask)))
                {
                    return getNodeEquality
                        (child1, child2).isAtLeast(TaskEquality.LEXICALLY_EQUAL);
                }
            }
        }
        else if (child2 == null) {
            return true;
        }
        
        return false;
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSyntacticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
        ITask child1 = ((IIteration) task1).getMarkedTask();
        ITask child2 = ((IIteration) task2).getMarkedTask();
        
        if (child1 != null) {
            if (child2 == null) {
                return false;
            }
            else {
                // iterations may have 3 different structures.
                // 1. they have one child, which is the iterated one
                // 2. they have a sequence of children, which is iterated
                // 3. they have a selection of different iterated variants (usually the variants
                //    are semantically equal)
                // ignore the type of the children but check them for equality.
                
                return getNodeEquality(child1, child2).isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL);
            }
        }
        else if (child2 == null) {
            return true;
        }
        
        return false;
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSemanticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSemanticallyEqual(ITask task1, ITask task2) {
        return compare(task1, task2).isAtLeast(TaskEquality.SEMANTICALLY_EQUAL);
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#compare(ITask, ITask)
     */
    @Override
    public TaskEquality compare(ITask task1, ITask task2) {
        ITask child1 = ((IIteration) task1).getMarkedTask();
        ITask child2 = ((IIteration) task2).getMarkedTask();

        // if both iterations do not have children, they are equal although this doesn't make sense
        if ((child1 == null) && (child2 == null)) {
            return TaskEquality.LEXICALLY_EQUAL;
        }
        else if ((child1 == null) || (child2 == null)) {
            return TaskEquality.UNEQUAL;
        }

        // iterations may have 3 different structures.
        // 1. they have one child, which is the iterated one
        // 2. they have a sequence of children, which is iterated
        // 3. they have a selection of different iterated variants (usually the variants are
        // semantically equal)
        //
        // the permutations of the three variants in combination must be checked

        // check if both tasks are the same variants of iterations and if their children are equal.
        // This condition matches, if both iterations are the same variants of iteration. I.e. three
        // combinations of the permutation are handled herewith.
        TaskEquality taskEquality = getNodeEquality(child1, child2);
        
        if (taskEquality != null) {
            return taskEquality;
        }

        // compare one iteration with a single task as a child and another one with a selection of
        // semantically equal tasks
        return selectionChildrenSemanticallyEqualNode(child1, child2);
        
        // all other combinations (i.e. sequence with single child and sequence with selection)
        // can not match
    }

    /**
     * <p>
     * compares two tasks with each other by calling the rule manager. If the rule manager returns
     * identity, then the returned equality is set to lexically equal. The reason is, that
     * the children of the iterations are compared and that therefore the distinct iterations
     * can be at most lexically equal.
     * </p>
     * 
     * @param child1 the first task to be compared
     * @param child2 the second task to be compared
     * 
     * @return the determined equality being at most lexical equality.
     */
    private TaskEquality getNodeEquality(ITask child1, ITask child2) {
        TaskEquality taskEquality = callRuleManager(child1, child2, null);

        if (taskEquality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL)) {
            // prevent, that identical is returned, because the iterations itself are not identical
            // although the iterated tasks are
            if (taskEquality == TaskEquality.IDENTICAL) {
                return TaskEquality.LEXICALLY_EQUAL;
            }
            else {
                return taskEquality;
            }
        }
        
        return TaskEquality.UNEQUAL;
    }

    /**
     * <p>
     * compares two tasks. One of them must be a selection, the other one can be any task.
     * The method returns a task equality that is not <code>NodeEquality.UNEQUAL</code>
     * if the other task is at least semantically equal to the children of the selection. It
     * returns more concrete equalities, if the equality between the other task and the children
     * of the selection is more concrete.
     * </p> 
     * 
     * @param task1 the first task to compare
     * @param task2 the second task to compare
     * 
     * @return as described
     */
    private TaskEquality selectionChildrenSemanticallyEqualNode(ITask task1, ITask task2) {
        ISelection selection = null;
        ITask task = null;
        if (task1 instanceof ISelection) {
            selection = (ISelection) task1;
            task = task2;
        }
        else if (task2 instanceof ISelection) {
            selection = (ISelection) task2;
            task = task1;
        }
        else {
            return TaskEquality.UNEQUAL;
        }

        // Iterations, where one has a selection and the other one not can at most be syntactically
        // equal but not identical
        TaskEquality commonDenominatorForAllComparisons = TaskEquality.SYNTACTICALLY_EQUAL;

        for (ITask child : selection.getChildren()) {
            TaskEquality taskEquality =
                  callRuleManager(task, child, commonDenominatorForAllComparisons);

            if ((taskEquality == null) || (taskEquality == TaskEquality.UNEQUAL))
            {
                return TaskEquality.UNEQUAL;
            }
            
            commonDenominatorForAllComparisons =
                commonDenominatorForAllComparisons.getCommonDenominator(taskEquality);
        }

        return commonDenominatorForAllComparisons;
    }

    /**
     * <p>
     * used to to call the task equality rule manager for the comparison of the two provided
     * children. If no required equality level is provided, than the most concrete equality is
     * returned. Otherwise, the required equality is returned as long as the children are equal
     * on that level.
     * </p> 
     * 
     * @param child1                the first task to be compared
     * @param child2                the second task to be compared
     * @param requiredEqualityLevel the equality level to be checked for
     * 
     * @return the determined equality
     */
    private TaskEquality callRuleManager(ITask        child1,
                                         ITask        child2,
                                         TaskEquality requiredEqualityLevel)
    {
        if (requiredEqualityLevel == null) {
            return mRuleManager.compare(child1, child2);
        }
        else if (mRuleManager.areAtLeastEqual(child1, child2, requiredEqualityLevel)) {
            return requiredEqualityLevel;
        }
        else {
            return TaskEquality.UNEQUAL;
        }
    }
}
