//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules.patterns;

import java.util.Arrays;
import java.util.List;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.usability.tasktree.IterativeDFSFilterStrategy;
import de.ugoe.cs.autoquest.usability.tasktree.filters.TaskModelFilter;
import de.ugoe.cs.autoquest.usability.tasktree.filters.TaskTypeFilter;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Alexander Deicke
 */
public class UsagePattern {
    
    private TaskModelFilter taskTreeFilter = new TaskModelFilter(new IterativeDFSFilterStrategy());
    
    private TaskTypeFilter concernedTask;

    private List<UsagePatternVisitor> patternVisitors;
    
    private boolean present = false;
    
    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param concernedNode
     * @param eventType
     */
    public UsagePattern(TaskTypeFilter concernedNode,
                        UsagePatternVisitor... patternVisitor)
    {
        this.patternVisitors = Arrays.asList(patternVisitor);
        this.concernedTask = concernedNode;
    }

    public boolean containedIn(ITaskModel taskModel) {
        List<ITask> allConcernedTasks = filterAllConcernedTasksFrom(taskModel);
        for(ITask concernedTask : allConcernedTasks) {
            checkTask(concernedTask);  
            if(this.present) break;
        }
        return this.present;
    }

    private void checkTask(ITask concernedTask) {
        applyAllVisitors(concernedTask);
        if(allVisitorsArePresent()) {
            this.present = true;
        } else {
            resetAllVisitors();
        }
    }
    
    public boolean containedIn(ITask task) {
        checkTask(task);
        return this.present;      
    }

    private void applyAllVisitors(ITask concernedTask) {
        Optional<UsagePatternVisitor> previousVisitor = Optional.absent();
        for(UsagePatternVisitor visitor : patternVisitors) {
            if(appliedOnSelectionNode(previousVisitor)) {
                for(ITask selection : previousVisitor.get().getRetainedSelectionNodes()) {
                    selection.accept(visitor);
                }
            } else {
                previousVisitor = Optional.of(visitor);
                concernedTask.accept(visitor);
            }
        }
    }

    private boolean appliedOnSelectionNode(Optional<UsagePatternVisitor> previousVisitor) {
        return previousVisitor.isPresent() && previousVisitor.get().hasExcludedSelectionNodes();
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskTree
     * @return
     */
    private List<ITask> filterAllConcernedTasksFrom(ITaskModel taskModel) {
        return this.taskTreeFilter.filterByNodeType(this.concernedTask).from(taskModel).tasksMatchedFilter();
    }
    
    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @return
     */
    private boolean allVisitorsArePresent() {
        Iterable<UsagePatternVisitor> allPresent = Iterables.filter(this.patternVisitors, new Predicate<UsagePatternVisitor>() {
            
            public boolean apply(UsagePatternVisitor visitor) {
                return visitor.isPresent();
            }
            
        });
        return Iterables.size(allPresent) == this.patternVisitors.size();
    }
    
    /**
     * <p>
     * TODO: comment
     * </p>
     *
     */
    private void resetAllVisitors() {
        for(UsagePatternVisitor visitor : this.patternVisitors) {
            visitor.reset();
        }
        
    }

}
