//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.manager;

import de.ugoe.cs.autoquest.tasktrees.taskequality.TaskEqualityRuleManager;
import de.ugoe.cs.autoquest.tasktrees.temporalrelation.TemporalRelationshipRuleManager;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskBuilder;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskFactory;
import de.ugoe.cs.autoquest.tasktrees.treeimpl.TaskBuilder;
import de.ugoe.cs.autoquest.tasktrees.treeimpl.TaskFactory;

/**
 * <p>
 * The component manager is the central reference for the distinct submodules required for
 * task tree generation. Such include the temporal relationship rule manager, the task equality
 * rule manager, the default task builder, as well as the default task factory.
 * </p>
 * 
 * @version 1.0
 * @author pharms
 */
public class ComponentManager {
    
    /**
     * <p>
     * singleton instance of this class
     * </p>
     */
    private static ComponentManager instance;

    /**
     * <p>
     * the default temporal relationship rule manager
     * </p>
     */
    private TemporalRelationshipRuleManager temporalRelationshipRuleManager;

    /**
     * <p>
     * the default task equality rule manager
     * </p>
     */
    private TaskEqualityRuleManager taskEqualityRuleManager;

    /**
     * <p>
     * the default task builder
     * </p>
     */
    private ITaskBuilder taskBuilder;

    /**
     * <p>
     * the default task factory
     * </p>
     */
    private ITaskFactory taskFactory;

    /**
     * <p>
     * returns the default temporal relationship rule manager
     * </p>
     * 
     * @return as described
     */
    public static TemporalRelationshipRuleManager getTemporalRelationshipRuleManager() {
        return getInstance().temporalRelationshipRuleManager;
    }

    /**
     * <p>
     * returns the default task equality rule manager
     * </p>
     * 
     * @return as described
     */
    public static TaskEqualityRuleManager getTaskEqualityRuleManager() {
        return getInstance().taskEqualityRuleManager;
    }

    /**
     * <p>
     * returns the default task builder
     * </p>
     * 
     * @return as described
     */
    public static ITaskBuilder getDefaultTaskBuilder() {
        return getInstance().taskBuilder;
    }

    /**
     * <p>
     * returns the default task factory
     * </p>
     * 
     * @return as described
     */
    public static ITaskFactory getDefaultTaskFactory() {
        return getInstance().taskFactory;
    }

    /**
     * <p>
     * clears the singleton instance. Needed for test purposes to ensure statelessness between
     * tests.
     * </p>
     */
    public static synchronized void clearInstance() {
        instance = null;
    }

    /**
     * <p>
     * returns the singleton instance of this class
     * </p>
     * 
     * @return as described
     */
    private static synchronized ComponentManager getInstance() {
        if (instance == null) {
            instance = new ComponentManager();
            instance.init();
        }
        return instance;
    }

    /**
     * <p>
     * initialized the component manager with all it default components which are the temporal
     * relationship rule manager, the task equality rule manager, the default task builder, as
     * well as the default task factory. 
     * </p>
     */
    private void init() {
        taskEqualityRuleManager = new TaskEqualityRuleManager();
        taskEqualityRuleManager.init();

        taskBuilder = new TaskBuilder();
        taskFactory = new TaskFactory();

        temporalRelationshipRuleManager = new TemporalRelationshipRuleManager
            (taskEqualityRuleManager, taskFactory, taskBuilder);
        temporalRelationshipRuleManager.init();
    }

}
