//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.treeifc;

/**
 * <p>
 * Builder for task models. Can be used to create and edit task models. May perform integrity
 * checks, though they may be incomplete as the integrity of a task model can not be ensured during
 * creation.
 * </p>
 */
public interface ITaskBuilder {

    /**
     * <p>
     * adds a child to a task instance. May ensure, that the child is a valid child considering
     * the task model of the parent. In that case, an IllegalArgumentException is thrown.
     * </p>
     * 
     * @param taskInstance the instance of add the child to
     * @param child        the child to be added
     * 
     * @throws IllegalArgumentException as described
     */
    void addChild(ITaskInstance taskInstance, ITaskInstance child) throws IllegalArgumentException;

    /**
     * <p>
     * adds a task instance to a user session
     * </p>
     *
     * @param session      the session to add the task instance to
     * @param taskInstance the task instance to add
     */
    void addExecutedTask(IUserSession session, ITaskInstance taskInstance);

    /**
     * <p>
     * adds a task instance to a task instance list
     * </p>
     * 
     * @param taskInstanceList the list to add the task instance to
     * @param taskInstance     the task instance to add
     */
    void addTaskInstance(ITaskInstanceList taskInstanceList, ITaskInstance taskInstance);

    /**
     * <p>
     * adds a task instance to a task instance list at a specific position. Subsequent task
     * instances will be moved one index forward
     * </p>
     * 
     * @param taskInstanceList the list to add the task instance to
     * @param index            the index of the task instance to add
     * @param taskInstance     the task instance to add
     * 
     * @throws IndexOutOfBoundsException if the index is invalid
     */
    void addTaskInstance(ITaskInstanceList taskInstanceList, int index, ITaskInstance taskInstance)
        throws IndexOutOfBoundsException;

    /**
     * <p>
     * sets a task instance in a task instance list at a specific position
     * </p>
     * 
     * @param taskInstanceList the list to set the task instance in
     * @param index            the index of the task instance to replace
     * @param taskInstance     the replacement for the task instance at the index
     * 
     * @throws IndexOutOfBoundsException if the index is invalid
     */
    void setTaskInstance(ITaskInstanceList taskInstanceList, int index, ITaskInstance taskInstance)
        throws IndexOutOfBoundsException;

    /**
     * <p>
     * sets the task model of a task instance
     * </p>
     * 
     * @param taskInstance the task instance to set the task model for
     * @param task         the task model of the instance
     */
    void setTask(ITaskInstance taskInstance, ITask task);

    /**
     * <p>
     * adds a child task to the end of a sequence
     * </p>
     * 
     * @param parent the sequence to add the child to
     * @param child  the child to be added
     */
    void addChild(ISequence parent, ITask child);

    /**
     * <p>
     * adds a child task to a specific index of a sequence
     * </p>
     * 
     * @param parent the sequence to add the child to
     * @param index  the index to set the child at
     * @param child  the child to be added
     * 
     * @throws IndexOutOfBoundsException if the index is invalid
     */
    void addChild(ISequence parent, int index, ITask child)
        throws IndexOutOfBoundsException;

    /**
     * <p>
     * replaces the child task of a sequence at a specific position
     * </p>
     * 
     * @param parent the sequence to replace the child in
     * @param index  the index to replace the child at
     * @param child  the child to be added
     * 
     * @throws IndexOutOfBoundsException if the index is invalid
     */
    void setChild(ISequence parent, int index, ITask child)
        throws IndexOutOfBoundsException;

    /**
     * <p>
     * adds a child task to a selection
     * </p>
     * 
     * @param parent the selection to add the child to
     * @param child  the child to be added
     */
    void addChild(ISelection parent, ITask child);

    /**
     * <p>
     * sets the child task of an iteration
     * </p>
     * 
     * @param parent the iteration to set the child of
     * @param child  the child to be set
     */
    void setMarkedTask(IIteration iteration, ITask child);

    /**
     * <p>
     * sets the child task of an optional
     * </p>
     * 
     * @param parent the optional to set the child of
     * @param child  the child to be set
     */
    void setMarkedTask(IOptional optional, ITask child);

    /**
     * <p>
     * removes the child of a sequence at a specific position
     * </p>
     * 
     * @param parent the sequence of which the child must be removed
     * @param index  the index of the child to be removed
     * 
     * @throws IndexOutOfBoundsException if the index is invalid
     */
    void removeChild(ISequence parent, int index)
        throws IndexOutOfBoundsException;

    /**
     * <p>
     * removes a child of a selection. Ignores the call, if the child is not found
     * (comparison using equals).
     * </p>
     * 
     * @param parent the selection of which the child must be removed
     * @param child  the child to be removes
     */
    void removeChild(ISelection parent, ITask child);

    /**
     * <p>
     * removes the entry of a task instance list at a specific position
     * </p>
     * 
     * @param taskInstanceList the task instance list of which the entry must be removed
     * @param index            the index of the entry to be removed
     * 
     * @throws IndexOutOfBoundsException if the index is invalid
     */
    void removeTaskInstance(ITaskInstanceList taskInstanceList, int index)
        throws IndexOutOfBoundsException;

    /**
     * <p>
     * replaces a child of a selection. Throws an IllegalArgumentException if the child is not
     * found (comparison using equals).
     * </p>
     * 
     * @param parent   the selection of which the child must be replace
     * @param oldChild the child to replace
     * @param newChild the replacement for the child
     * 
     * @throws as described
     */
    void replaceChild(ISelection parent, ITask oldChild, ITask newChild);

    /**
     * <p>
     * sets the description of a task
     * </p>
     * 
     * @param task        the task to set the description of
     * @param description the new description of the task
     */
    void setDescription(ITask task, String description);

}
