//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;

/**
 * <p>
 * This rule identifies two tasks as lexically equal, if they are both event tasks and
 * if their respective event types and targets equal. 
 * </p>
 * 
 * @author Patrick Harms
 */
public class EventTaskComparisonRule implements TaskComparisonRule {
    
    /* (non-Javadoc)
     * @see NodeComparisonRule#isApplicable(ITask, ITask)
     */
    @Override
    public boolean isApplicable(ITask task1, ITask task2) {
        return (task1 instanceof IEventTask) && (task2 instanceof IEventTask);
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areLexicallyEqual(ITask, ITask)
     */
    @Override
    public boolean areLexicallyEqual(ITask task1, ITask task2) {
        IEventTask eventTask1 = (IEventTask) task1;
        IEventTask eventTask2 = (IEventTask) task2;
        
        return (eventTask1.getEventType().equals(eventTask2.getEventType()) &&
                eventTask1.getEventTarget().equals(eventTask2.getEventTarget()));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSyntacticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
        return areLexicallyEqual(task1, task2);
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSemanticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSemanticallyEqual(ITask task1, ITask task2) {
        return areLexicallyEqual(task1, task2);
    }

    @Override
    public TaskEquality compare(ITask task1, ITask task2) {
        if (areLexicallyEqual(task1, task2)) {
            return TaskEquality.LEXICALLY_EQUAL;
        }
        else {
            return TaskEquality.UNEQUAL;
        }
    }

}
