//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules.metrics;

import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.absent;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.highRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.infoRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.lowRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.mediumRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.present;
import static org.fest.assertions.api.Assertions.assertThat;

import org.junit.Test;

import com.google.common.base.Optional;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.usability.result.UsabilityDefect;
import de.ugoe.cs.autoquest.usability.testutil.GenerateTaskModelUtil;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Alexander Deicke
 */
public class TextInputEntryRepetitionsEvaluatorTest {

    @Test
    public void should_return_no_recommendation() {
        // Given
        String spec = "Sequence {" + "  TextInput () {}" + "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(absent());
    }

    @Test
    public void should_return_recommendation_with_info_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a b c) {}" + "  Sequence {" + "    TextInput (a) {}"
                + "    TextInput (d) {}" + "    TextInput (e) {}" + "  }" + "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(infoRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_low_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a b c) {}" + "  Sequence {" + "    TextInput (a) {}"
                + "    TextInput (b) {}" + "    TextInput (c) {}" + "  }" + "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(lowRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_medium_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a b c d e f g h i j k l m) {}" + "  Sequence {"
                + "    TextInput (a) {}" + "    TextInput (b) {}" + "    TextInput (c) {}"
                + "    TextInput (d) {}" + "  }" + "  Iteration {" + "    TextInput (e) {}" + "  }"
                + "  TextInput (a) {}" + "  Selection {" + "    TextInput (b) {}"
                + "    TextInput (c) {}" + "    TextInput (d) {}" + "    TextInput (e) {}" + "  }"
                + "  Sequence {" + "    TextInput (a) {}" + "    Sequence {"
                + "      TextInput (b) {}" + "      TextInput (c) {}" + "      TextInput (d) {}"
                + "      TextInput (e) {}" + "    }" + "  }" + "  TextInput (f) {}" + "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(mediumRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_high_severity_level() {
        // Given
        String spec =
            "Sequence {" + "  TextInput (a b c) {}" + "  Sequence {" + "    TextInput (a) {}"
                + "    TextInput (b) {}" + "    TextInput (c) {}" + "    TextInput (a) {}" + "  }"
                + "  Iteration {" + "    TextInput (a) {}" + "  }" + "  TextInput (a) {}"
                + "  Selection {" + "    TextInput (b c) {}" + "    TextInput (a) {}"
                + "    TextInput (a c) {}" + "    TextInput (b a) {}" + "  }" + "  Sequence {"
                + "    TextInput (b c) {}" + "    Sequence {" + "      TextInput (d a c) {}"
                + "      TextInput (b b b a) {}" + "      TextInput (a a c c) {}"
                + "      TextInput (b b a) {}" + "    }" + "  }" + "  TextInput (d) {}" + "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityDefect> recommendation = new TextInputEntryRepetitionsMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(highRecommendationSeverityLevel());
    }
}
