//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;

/**
 * <p>
 * This class is capable of comparing any task which is not a selection with a
 * selection. This is needed, because selections may contain exactly that task. Therefore, if
 * this task is selected out of a selection the selection is equal to the task itself. 
 * The rule returns lexically equal, if the selection contains a lexically equal task. The same
 * applies for syntactical and semantical equality.
 * </p>

 * @author Patrick Harms
 */
public class TaskAndSelectionComparisonRule implements TaskComparisonRule {
    
    /* (non-Javadoc)
     * @see NodeComparisonRule#isApplicable(ITask, ITask)
     */
    @Override
    public boolean isApplicable(ITask task1, ITask task2) {
        return ((task1 instanceof ISelection) && (!(task2 instanceof ISelection))) ||
               ((task2 instanceof ISelection) && (!(task1 instanceof ISelection)));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areLexicallyEqual(ITask, ITask)
     */
    @Override
    public boolean areLexicallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.LEXICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSyntacticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.SYNTACTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#areSemanticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSemanticallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.SEMANTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see NodeComparisonRule#compare(ITask, ITask)
     */
    @Override
    public TaskEquality compare(ITask task1, ITask task2) {
        return getEquality(task1, task2, null);
    }
    
    /**
     * <p>
     * compares two tasks with each other checking for the provided required level of
     * equality. One of the tasks must be a selection, the other one not. If this is not the
     * case, the method returns null. The returned equality level is at most lexical equality
     * as the selection can not be identical to something not being a selection.
     * </p>
     * 
     * @param task1                 the first task to be compared
     * @param task2                 the second task to be compared
     * @param requiredEqualityLevel the equality level to be checked for
     * 
     * @return the determined equality.
     */
    private TaskEquality getEquality(ITask task1, ITask task2, TaskEquality requiredEqualityLevel) {
        ISelection selection = null;
        ITask task = null;
        
        if (task1 instanceof ISelection) {
            if (task2 instanceof ISelection) {
                // the rule is not responsible for two selections
                return null;
            }
            
            selection = (ISelection) task1;
            task = task2;
        }
        else if (task2 instanceof ISelection) {
            if (task1 instanceof ISelection) {
                // the rule is not responsible for two selections
                return null;
            }
            
            selection = (ISelection) task2;
            task = task1;
        }
        else {
            return null;
        }

        // now, that we found the selection and the task, lets compare the children of the selection
        // with the task.
        List<ITask> children = selection.getChildren();
        
        if (children.size() < 1) {
            return null;
        }

        TaskEquality mostConcreteNodeEquality = null;
        
        for (ITask child : children) {
            TaskEquality taskEquality = callRuleManager(child, task, requiredEqualityLevel);
            
            if (taskEquality != TaskEquality.UNEQUAL) {
                if (mostConcreteNodeEquality == null) {
                    mostConcreteNodeEquality = taskEquality;
                }
                else if (mostConcreteNodeEquality.isAtLeast(taskEquality)) {
                    mostConcreteNodeEquality = taskEquality;
                    
                }
                
                if ((requiredEqualityLevel != null) &&
                    (mostConcreteNodeEquality.isAtLeast(requiredEqualityLevel)))
                {
                    // if we found one child of the selection that is as equal as required, then
                    // we can consider the selection to be sufficiently equal to the other task.
                    // So we break up checking further children.
                    break;
                }
            }
        }
        
        // although the subtask may be identical to the task, we can not return identical, as
        // the selection is not identical to the task, but at most lexically equal
        if (mostConcreteNodeEquality == TaskEquality.IDENTICAL) {
            return TaskEquality.LEXICALLY_EQUAL;
        }
        else {
            return mostConcreteNodeEquality;
        }

    }
    
    /**
     * <p>
     * used to to call the task equality rule manager for the comparison of the two provided
     * children. If no required equality level is provided, than the most concrete equality is
     * returned. Otherwise, the required equality is returned as long as the children are equal
     * on that level.
     * </p> 
     * 
     * @param child1                the first task to be compared
     * @param child2                the second task to be compared
     * @param requiredEqualityLevel the equality level to be checked for
     * 
     * @return the determined equality
     */
    private TaskEquality callRuleManager(ITask        child1,
                                         ITask        child2,
                                         TaskEquality requiredEqualityLevel)
    {
        if (requiredEqualityLevel == null) {
            return TaskEqualityRuleManager.getInstance().compare(child1, child2);
        }
        else if (TaskEqualityRuleManager.getInstance().areAtLeastEqual
                     (child1, child2, requiredEqualityLevel))
        {
            return requiredEqualityLevel;
        }
        else {
            return TaskEquality.UNEQUAL;
        }
    }
}
