//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.result;

import java.util.Collections;
import java.util.EnumMap;
import java.util.Map;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.Maps;

/**
 * <p>
 * Description for a usability problem.
 * </p>
 * 
 * @author Alexander Deicke
 */
public class UsabilityProblemDescription {
    /**
     * <p>
     * The severity of the defect.
     * </p>
     */

    private UsabilityProblemSeverityLevel severityLevel = UsabilityProblemSeverityLevel.NONE;

    /**
     * <p>
     * A detailed description of the defect.
     * </p>
     */
    private String description;

    /**
     * <p>
     * Assignment of all possible severity values to a certain threshold. This is used, to determine
     * the severity level during the usability evaluation.
     * </p>
     */
    private final EnumMap<UsabilityProblemSeverityLevel, Double> severity;

    /**
     * <p>
     * Constructor. Creates a new UsabilityDefectDescription for given description and all possible
     * severity level.
     * </p>
     * 
     * @param description
     *            description of the defect
     * @param severity
     *            all possible severity level and their threshold
     */
    public UsabilityProblemDescription(String description,
                                      EnumMap<UsabilityProblemSeverityLevel, Double> severity)
    {
        this.description = description;
        this.severity = severity;
    }

    /**
     * <p>
     * Gets the severity level of this defect.
     * </p>
     * 
     * @return severity level of this defect
     */
    public UsabilityProblemSeverityLevel getSeverityLevel() {
        return this.severityLevel;
    }

    /**
     * <p>
     * Checks, if a defect is present. Therefore it uses a value, which is used to determine the
     * severity level.
     * </p>
     * 
     * @param evaluationMetric
     *            value, which determines the severity level of a defect
     * @return iff defect is present, a {@linkplain UsabilityProblemDescription} of the defect
     */
    public Optional<UsabilityProblemDescription> isPresent(float evaluationMetric) {
        Optional<UsabilityProblemDescription> defect = Optional.absent();
        Map<UsabilityProblemSeverityLevel, Double> matchingSeverityLevels =
            Maps.filterValues(this.severity, severityForMetric(evaluationMetric));
        if (matchingSeverityLevels.isEmpty()) {
            return defect;
        }
        setDefectAttributes(matchingSeverityLevels, evaluationMetric);
        return Optional.of(this);
    }

    /**
     * <p>
     * Initializes the properties of a {@linkplain UsabilityProblemDescription}.
     * </p>
     * 
     * @param matchingSeverityLevels
     *            all severity level, where {@linkplain evaluationMetric} exceeds the threshold
     * @param evaluationMetric
     *            measure for the defect
     */
    private void setDefectAttributes(Map<UsabilityProblemSeverityLevel, Double> matchingSeverityLevels,
                                     float evaluationMetric)
    {
        BiMap<Double, UsabilityProblemSeverityLevel> inverse =
            HashBiMap.create(matchingSeverityLevels).inverse();
        this.severityLevel = inverse.get(Collections.max(inverse.keySet()));
        this.description = String.format(this.description, evaluationMetric);
    }

    /**
     * <p>
     * Gets severity level depending on a measurement for the defect.
     * </p>
     * 
     * @param evaluationMetric
     *            measure for the defect
     * 
     * @return severity level, if measurement is equal or bigger as the threshold of the severity
     *         level
     */
    private Predicate<Double> severityForMetric(final float evaluationMetric) {
        return new Predicate<Double>() {

            public boolean apply(Double severityThreshold) {
                return evaluationMetric == severityThreshold ||
                    evaluationMetric > severityThreshold;
            }

        };
    }

}
