//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules.metrics;

import static de.ugoe.cs.autoquest.usability.taskmodel.filter.types.EventTypeFilter.TEXT_INPUT;

import java.util.List;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.usability.EvaluationMethodCaller;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescriptionResolver;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescription;
import de.ugoe.cs.autoquest.usability.rules.UsabilityMetric;
import de.ugoe.cs.autoquest.usability.rules.UsabilityRule;
import de.ugoe.cs.autoquest.usability.taskmodel.filter.FilterResult;
import de.ugoe.cs.autoquest.usability.taskmodel.filter.IterativeDFSFilterStrategy;
import de.ugoe.cs.autoquest.usability.taskmodel.filter.types.TaskModelFilter;

/**
 * <p>
 * Metric, which measures the ratio between the text input and the non text input events.
 * </p>
 * 
 * @author Alexander Deicke
 */
public class TextInputRatioMetric extends UsabilityRule implements UsabilityMetric {

    /**
     * <p>
     * Constructor. Creates a new {@code TextInputRatioMetric} for a given task model.
     * </p>
     * 
     * @param taskTree
     */
    public TextInputRatioMetric(ITaskModel taskModel) {
        super(taskModel);
        this.name = "TextInputRatio";
        this.defect =
            new UsabilityProblemDescriptionResolver().descriptionFor(this.getClass()
                .getSimpleName());
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.usability.rules.UsabilityRule#check()
     */
    @Override
    public Optional<UsabilityProblemDescription> calculate() {
        FilterResult textInputEvents = extractNodesFromTaskTree();
        float evaluationMetric =
            calculateEvaluationMetric(textInputEvents.tasksMatchedFilter(),
                                      textInputEvents.tasksNotMatchedFilter());
        return this.defect.isPresent(evaluationMetric);
    }

    /**
     * 
     * <p>
     * Filters all text input events from task model.
     * </p>
     * 
     * @return {@code FilterResult}
     */
    private FilterResult extractNodesFromTaskTree() {
        return new TaskModelFilter(new IterativeDFSFilterStrategy()).filterByEventType(TEXT_INPUT)
            .from(this.taskModel);
    }

    /**
     * 
     * <p>
     * Calculates the metric.
     * </p>
     * 
     * @param textInputEvents
     *            all text input events
     * @param nonTextInputEvents
     *            all non text input events
     * @return ratio between text input and non text input events
     */
    private float calculateEvaluationMetric(List<ITask> textInputEvents,
                                            List<ITask> nonTextInputEvents)
    {
        float numberOfTextInputEvents = textInputEvents.size();
        float numberOfNonTextInputEvents = nrOfEventTasksNotMatchedFilter(nonTextInputEvents);
        return numberOfTextInputEvents / (numberOfTextInputEvents + numberOfNonTextInputEvents);
    }

    /**
     * 
     * <p>
     * Filters all {@link IEventTask}s from non text input event.
     * </p>
     * 
     * @param nonTextInputEvents
     *            all non text input events
     * @return number of {@link IEventTask}s
     */
    private int nrOfEventTasksNotMatchedFilter(List<ITask> nonTextInputEvents) {
        return Iterables.size(Iterables.filter(nonTextInputEvents, new Predicate<ITask>() {

            @Override
            public boolean apply(ITask task) {
                return task instanceof IEventTask;
            }
        }));
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * de.ugoe.cs.autoquest.usability.rules.UsabilityRule#callEvaluationMetho(de.ugoe.cs.autoquest
     * .usability.EvaluationMethodCaller)
     */
    @Override
    public Optional<UsabilityProblemDescription> callEvaluationMethod(EvaluationMethodCaller evaluationMethodCaller)
    {
        return evaluationMethodCaller.check(this);
    }
}
