//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.treeimpl;

import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IOptional;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskBuilder;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstanceList;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IUserSession;

/**
 * <p>
 * this is the default implementation of the interface {@link ITaskBuilder}. It
 * does not do anything fancy except implementing the interface. In some situations, it performs
 * a check if the model or instances to be created a valid. However, this can not be done
 * in any situation of the creation process.
 * </p>
 * 
 * @author Patrick Harms
 */
public class TaskBuilder implements ITaskBuilder {

    /* (non-Javadoc)
     * @see ITaskBuilder#addChild(ITaskInstance,ITaskInstance)
     */
    @Override
    public void addChild(ITaskInstance parent, ITaskInstance child) throws IllegalArgumentException
    {
        if (!(parent instanceof TaskInstance)) {
            throw new IllegalArgumentException
                ("illegal type of task instance provided: " + parent.getClass());
        }

        if (!(child instanceof TaskInstance)) {
            throw new IllegalArgumentException
                ("illegal type of task instance provided: " + parent.getClass());
        }
        
        // check, that the correct number of children for the distinct types are added
        ITask task = parent.getTask();
        
        if (task instanceof IEventTask) {
            throw new IllegalArgumentException
                ("can not add children to a task instance of an event task");
        }
        else if (task instanceof ISelection) {
            if (parent.getChildren().size() > 0) {
                throw new IllegalArgumentException
                    ("the instance of a selection must have at most one child");
            }
        }
        else if (task instanceof IOptional) {
            if (parent.getChildren().size() > 1) {
                throw new IllegalArgumentException
                    ("the instance of an optional must have at most one child");
            }
        }
        /*else if (task instanceof IIteration) {
            for (ITaskInstance childInstance : parent.getChildren()) {
                if (!childInstance.getTask().equals(child.getTask())) {
                    throw new IllegalArgumentException
                        ("all children of an instance of an iteration must have exactly the " +
                         "same type");
                }
            }
        }
        
        boolean foundChildTask = false;
        if (parent.getTask() instanceof IStructuringTemporalRelationship) {
            IStructuringTemporalRelationship parentTask =
                (IStructuringTemporalRelationship) parent.getTask();
        
            for (ITask parentTaskChild : parentTask.getChildren()) {
                if (parentTaskChild.equals(child.getTask())) {
                    foundChildTask = true;
                    break;
                }
            }
        }
        else if (parent.getTask() instanceof IMarkingTemporalRelationship) {
            IMarkingTemporalRelationship parentTask =
                (IMarkingTemporalRelationship) parent.getTask();
            
            foundChildTask = parentTask.getMarkedTask() != null ?
                parentTask.getMarkedTask().equals(child.getTask()) : false;
        }
        
        if (!foundChildTask) {
            throw new IllegalArgumentException
                ("the task of the child instance to be added does not belong to the children " +
                 "of the task of the parent instance");
        }*/

        // finally, after all checks are positive, add the child
        ((TaskInstance) parent).addChild(child);
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#addExecutedTask(IUserSession, ITaskInstance)
     */
    @Override
    public void addExecutedTask(IUserSession session, ITaskInstance taskInstance) {
        if (!(session instanceof UserSession)) {
            throw new IllegalArgumentException
                ("illegal type of session provided: " + session.getClass());
        }

        if (!(taskInstance instanceof TaskInstance)) {
            throw new IllegalArgumentException
                ("illegal type of task instance provided: " + taskInstance.getClass());
        }
        
        ((UserSession) session).addExecutedTask(taskInstance);
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#addTaskInstance(ITaskInstanceList, ITaskInstance)
     */
    @Override
    public void addTaskInstance(ITaskInstanceList taskInstanceList, ITaskInstance taskInstance) {
        if (taskInstanceList instanceof TaskInstance) {
            ((TaskInstance) taskInstanceList).addChild(taskInstance);
        }
        else if (taskInstanceList instanceof UserSession) {
            ((UserSession) taskInstanceList).addExecutedTask(taskInstance);
        }
        else {
            throw new IllegalArgumentException
                ("illegal type of task instance list provided: " + taskInstanceList.getClass());
        }
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#addTaskInstance(ITaskInstanceList, int, ITaskInstance)
     */
    @Override
    public void addTaskInstance(ITaskInstanceList taskInstanceList,
                                int               index,
                                ITaskInstance     taskInstance)
    {
        if (taskInstanceList instanceof TaskInstance) {
            ((TaskInstance) taskInstanceList).addChild(index, taskInstance);
        }
        else if (taskInstanceList instanceof UserSession) {
            ((UserSession) taskInstanceList).addExecutedTask(index, taskInstance);
        }
        else {
            throw new IllegalArgumentException
                ("illegal type of task instance list provided: " + taskInstanceList.getClass());
        }
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#setTaskInstance(ITaskInstanceList, int, ITaskInstance)
     */
    @Override
    public void setTaskInstance(ITaskInstanceList taskInstanceList,
                                int               index,
                                ITaskInstance     taskInstance)
    {
        if (taskInstanceList instanceof TaskInstance) {
            ((TaskInstance) taskInstanceList).removeChild(index);
            ((TaskInstance) taskInstanceList).addChild(index, taskInstance);
        }
        else if (taskInstanceList instanceof UserSession) {
            ((UserSession) taskInstanceList).removeExecutedTask(index);
            ((UserSession) taskInstanceList).addExecutedTask(index, taskInstance);
        }
        else {
            throw new IllegalArgumentException
                ("illegal type of task instance list provided: " + taskInstanceList.getClass());
        }
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#setTask(ITaskInstance, ITask)
     */
    @Override
    public void setTask(ITaskInstance taskInstance, ITask task) {
        if (!(taskInstance instanceof TaskInstance)) {
            throw new IllegalArgumentException
                ("illegal type of task instance provided: " + taskInstance.getClass());
        }
        
        ((TaskInstance) taskInstance).setTask(task);
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#addChild(ISequence, ITask)
     */
    @Override
    public void addChild(ISequence parent, ITask child) {
        if (!(parent instanceof Sequence)) {
            throw new IllegalArgumentException
                ("illegal type of sequence provided: " + parent.getClass());
        }

        addChildInternal((Sequence) parent, -1, child);
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#addChild(ISequence, int, ITask)
     */
    @Override
    public void addChild(ISequence parent, int index, ITask child) {
        if (!(parent instanceof Sequence)) {
            throw new IllegalArgumentException
                ("illegal type of sequence provided: " + parent.getClass());
        }

        addChildInternal((Sequence) parent, index, child);
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#setChild(ISequence, int, ITask)
     */
    @Override
    public void setChild(ISequence parent, int index, ITask child) {
        if (!(parent instanceof Sequence)) {
            throw new IllegalArgumentException
                ("illegal type of sequence provided: " + parent.getClass());
        }

        ((Sequence) parent).removeChild(index);
        addChildInternal((Sequence) parent, index, child);
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#addChild(ISelection, ITask)
     */
    @Override
    public void addChild(ISelection parent, ITask child) {
        if (!(parent instanceof Selection)) {
            throw new IllegalArgumentException
                ("illegal type of selection provided: " + parent.getClass());
        }

        addChildInternal((Selection) parent, -1, child);
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#setMarkedTask(IIteration, ITask)
     */
    @Override
    public void setMarkedTask(IIteration iteration, ITask newChild) {
        if (!(iteration instanceof Iteration)) {
            throw new IllegalArgumentException
                ("illegal type of iteration provided: " + iteration.getClass());
        }

        if (!(newChild instanceof Task)) {
            throw new IllegalArgumentException
                ("illegal type of task provided: " + newChild.getClass());
        }

        ((Iteration) iteration).setMarkedTask(newChild);
    }

    /* (non-Javadoc)
     * @see ITaskTreeBuilder#setChild(IOptional, ITaskTreeNode)
     */
    @Override
    public void setMarkedTask(IOptional optional, ITask newChild) {
        if (!(optional instanceof Optional)) {
            throw new IllegalArgumentException
                ("illegal type of optional provided: " + optional.getClass());
        }

        if (!(newChild instanceof Task)) {
            throw new IllegalArgumentException
                ("illegal type of task provided: " + newChild.getClass());
        }

        ((Optional) optional).setMarkedTask(newChild);
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#removeChild(ISequence, int)
     */
    @Override
    public void removeChild(ISequence parent, int index) {
        if (!(parent instanceof Sequence)) {
            throw new IllegalArgumentException
                ("illegal type of sequence provided: " + parent.getClass());
        }

        ((Sequence) parent).removeChild(index);
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#removeChild(ISelection, ITask)
     */
    @Override
    public void removeChild(ISelection parent, ITask child) {
        if (!(parent instanceof Selection)) {
            throw new IllegalArgumentException
                ("illegal type of selection provided: " + parent.getClass());
        }

        List<ITask> children = parent.getChildren();
        
        for (int i = 0; i < children.size(); i++) {
            if ((children.get(i) == child) ||
                ((children.get(i) != null) && (children.get(i).equals(child))))
            {
                ((Selection) parent).removeChild(i);
                break;
            }
        }
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#removeTaskInstance(ITaskInstanceList, int)
     */
    @Override
    public void removeTaskInstance(ITaskInstanceList taskInstanceList, int index) {
        if (taskInstanceList instanceof TaskInstance) {
            ((TaskInstance) taskInstanceList).removeChild(index);
        }
        else if (taskInstanceList instanceof UserSession) {
            ((UserSession) taskInstanceList).removeExecutedTask(index);
        }
        else {
            throw new IllegalArgumentException
                ("illegal type of task instance list provided: " + taskInstanceList.getClass());
        }
    }

    /* (non-Javadoc)
     * @see ITaskTreeBuilder#replaceChild(ISelection, ITaskTreeNode, ITaskTreeNode)
     */
    @Override
    public void replaceChild(ISelection parent, ITask oldChild, ITask newChild) {
        if (!(parent instanceof Selection)) {
            throw new IllegalArgumentException
                ("illegal type of selection provided: " + parent.getClass());
        }

        List<ITask> children = parent.getChildren();
        
        for (int i = 0; i < children.size(); i++) {
            if ((children.get(i) == oldChild) ||
                ((children.get(i) != null) && (children.get(i).equals(oldChild))))
            {
                ((Selection) parent).removeChild(i);
                ((Selection) parent).addChild(i, newChild);
                break;
            }
        }
    }

    /* (non-Javadoc)
     * @see ITaskBuilder#setDescription(ITask, java.lang.String)
     */
    @Override
    public void setDescription(ITask parent, String description) {
        if (!(parent instanceof Task)) {
            throw new IllegalArgumentException
                ("illegal type of task provided: " + parent.getClass());
        }

        ((Task) parent).setDescription(description);
    }

    /**
     * <p>
     * internal convenience method for adding children to a structuring temporal relationship
     * including a check for the child type.
     * </p>
     */
    private void addChildInternal(StructuringTemporalRelationship parent, int index, ITask child) {
        if (!(child instanceof Task)) {
            throw new IllegalArgumentException
                ("illegal type of task provided: " + child.getClass());
        }

        if (index > -1) {
            parent.addChild(index, child);
        }
        else {
            parent.addChild(child);
        }
    }

}
