//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import static org.junit.Assert.*;

import org.junit.Test;

import de.ugoe.cs.autoquest.eventcore.IEventTarget;
import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.eventcore.StringEventType;
import de.ugoe.cs.autoquest.tasktrees.taskequality.EventTaskComparisonRule;
import de.ugoe.cs.autoquest.tasktrees.taskequality.TaskEquality;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskFactory;
import de.ugoe.cs.autoquest.tasktrees.treeimpl.TaskFactory;
import de.ugoe.cs.autoquest.test.DummyGUIElement;

/**
 * @author Patrick Harms
 */
public class EventTaskComparisonRuleTest {

    /**
     *
     */
    @Test
    public void test_isApplicable_01() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        assertTrue(rule.isApplicable(task1, task1));
    }
    
    /**
     *
     */
    @Test
    public void test_isApplicable_02() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        IEventType eventType2 = new StringEventType("eventType2");
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);

        assertTrue(rule.isApplicable(task1, task2));
        assertTrue(rule.isApplicable(task2, task1));
    }
   
    /**
     *
     */
    @Test
    public void test_isApplicable_03() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
       
        ITask selection = taskFactory.createNewSelection();

        assertFalse(rule.isApplicable(task1, selection));
        assertFalse(rule.isApplicable(selection, task1));
    }
    
    /**
     *
     */
    @Test
    public void test_isApplicable_04() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        ITask sequence = taskFactory.createNewSequence();

        assertFalse(rule.isApplicable(task1, sequence));
        assertFalse(rule.isApplicable(sequence, task1));
    }
    
    /**
     *
     */
    @Test
    public void test_isApplicable_05() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        ITask iteration = taskFactory.createNewIteration();

        assertFalse(rule.isApplicable(task1, iteration));
        assertFalse(rule.isApplicable(iteration, task1));
    }
    
    /**
     *
     */
    @Test
    public void test_isApplicable_06() {
        ITaskFactory taskFactory = new TaskFactory();

        EventTaskComparisonRule rule = new EventTaskComparisonRule();

        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);

        ITask optional = taskFactory.createNewOptional();

        assertFalse(rule.isApplicable(task1, optional));
        assertFalse(rule.isApplicable(optional, task1));
    }
    
    /**
     *
     */
    @Test
    public void test_compare_01() {
        ITaskFactory taskFactory = new TaskFactory();
        
        EventTaskComparisonRule rule = new EventTaskComparisonRule();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(task1, task1));
        assertTrue(rule.areLexicallyEqual(task1, task1));
        assertTrue(rule.areSyntacticallyEqual(task1, task1));
        assertTrue(rule.areSemanticallyEqual(task1, task1));
    }
    
    /**
     *
     */
    @Test
    public void test_compare_02() {
        ITaskFactory taskFactory = new TaskFactory();
        
        EventTaskComparisonRule rule = new EventTaskComparisonRule();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        ITask task2 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(task1, task2));
        assertTrue(rule.areLexicallyEqual(task1, task2));
        assertTrue(rule.areSyntacticallyEqual(task1, task2));
        assertTrue(rule.areSemanticallyEqual(task1, task2));

        assertEquals(TaskEquality.LEXICALLY_EQUAL, rule.compare(task2, task1));
        assertTrue(rule.areLexicallyEqual(task2, task1));
        assertTrue(rule.areSyntacticallyEqual(task2, task1));
        assertTrue(rule.areSemanticallyEqual(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_compare_03() {
        ITaskFactory taskFactory = new TaskFactory();
        
        EventTaskComparisonRule rule = new EventTaskComparisonRule();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new StringEventType("eventType2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget1);

        assertEquals(TaskEquality.UNEQUAL, rule.compare(task1, task2));
        assertFalse(rule.areLexicallyEqual(task1, task2));
        assertFalse(rule.areSyntacticallyEqual(task1, task2));
        assertFalse(rule.areSemanticallyEqual(task1, task2));

        assertEquals(TaskEquality.UNEQUAL, rule.compare(task2, task1));
        assertFalse(rule.areLexicallyEqual(task2, task1));
        assertFalse(rule.areSyntacticallyEqual(task2, task1));
        assertFalse(rule.areSemanticallyEqual(task2, task1));
    }

    /**
     *
     */
    @Test
    public void test_compare_04() {
        ITaskFactory taskFactory = new TaskFactory();
        
        EventTaskComparisonRule rule = new EventTaskComparisonRule();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType1, eventTarget2);
        
        assertEquals(TaskEquality.UNEQUAL, rule.compare(task1, task2));
        assertFalse(rule.areLexicallyEqual(task1, task2));
        assertFalse(rule.areSyntacticallyEqual(task1, task2));
        assertFalse(rule.areSemanticallyEqual(task1, task2));

        assertEquals(TaskEquality.UNEQUAL, rule.compare(task2, task1));
        assertFalse(rule.areLexicallyEqual(task2, task1));
        assertFalse(rule.areSyntacticallyEqual(task2, task1));
        assertFalse(rule.areSemanticallyEqual(task2, task1));
    }


    /**
     *
     */
    @Test
    public void test_compare_05() {
        ITaskFactory taskFactory = new TaskFactory();
        
        EventTaskComparisonRule rule = new EventTaskComparisonRule();
        
        IEventType eventType1 = new StringEventType("eventType1");
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        ITask task1 = taskFactory.createNewEventTask(eventType1, eventTarget1);
        
        IEventType eventType2 = new StringEventType("eventType2");
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        ITask task2 = taskFactory.createNewEventTask(eventType2, eventTarget2);
        
        assertEquals(TaskEquality.UNEQUAL, rule.compare(task1, task2));
        assertFalse(rule.areLexicallyEqual(task1, task2));
        assertFalse(rule.areSyntacticallyEqual(task1, task2));
        assertFalse(rule.areSemanticallyEqual(task1, task2));

        assertEquals(TaskEquality.UNEQUAL, rule.compare(task2, task1));
        assertFalse(rule.areLexicallyEqual(task2, task1));
        assertFalse(rule.areSyntacticallyEqual(task2, task1));
        assertFalse(rule.areSemanticallyEqual(task2, task1));
    }

}
