//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usageprofiles;

import static org.junit.Assert.*;

import java.util.Iterator;

import org.junit.Test;

/**
 * <p>
 * The class <code>SymbolMapTest</code> contains tests for the class
 * <code>{@link SymbolMap}</code>.
 * </p>
 * 
 * @author Patrick Harms
 */
public class SymbolMapTest {

    @Test
    public void testSymbolMap_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        assertNotNull(symbolMap);
        assertEquals(0, symbolMap.size());
    }

    @Test
    public void testSymbolMap_2() {
        SymbolMap<String, String> symbolMap1 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        symbolMap1.addSymbol("symbol", "value");
        
        SymbolMap<String, String> symbolMap2 = new SymbolMap<String, String>(symbolMap1);
        
        assertNotNull(symbolMap2);
        assertSymbolMapEntries(symbolMap2, new String[] { "symbol" } , new String[] { "value" });
    }

    @Test(expected = java.lang.IllegalArgumentException.class)
    public void testSymbolMap_3() throws Exception {
        new SymbolMap<String, String>((DefaultSymbolComparator<String>) null);
    }

    @Test(expected = java.lang.IllegalArgumentException.class)
    public void testSymbolMap_4() throws Exception {
        new SymbolMap<String, String>((SymbolMap<String, String>) null);
    }
    
    @Test
    public void testAddSymbol_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", "value1");
        
        assertNotNull(symbolMap);
        assertSymbolMapEntries(symbolMap, new String[] { "symbol1" } , new String[] { "value1" });
    }
    
    @Test
    public void testAddSymbol_2() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        String[] symbols = new String[entryCount];
        String[] values = new String[entryCount];
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
            symbols[i] = "symbol" + i;
            values[i] = "value" + i;
        }
        
        assertNotNull(symbolMap);
        assertSymbolMapEntries(symbolMap, symbols, values);
    }
    
    @Test
    public void testAddSymbol_3() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        String[] symbols = new String[entryCount];
        String[] values = new String[entryCount];
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap.addSymbol("symbol" + i, "value" + i);
                symbols[i] = "symbol" + i;
                values[i] = "value" + i;
            }
            else {
                symbolMap.addSymbol("symbol" + i, null);
                symbols[i] = "symbol" + i;
                values[i] = null;
            }
        }
        
        assertNotNull(symbolMap);
        assertSymbolMapEntries(symbolMap, symbols, values);
    }
    
    @Test(expected = java.lang.IllegalArgumentException.class)
    public void testAddSymbol_4() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol(null, null);
    }    
    
    @Test
    public void testSize_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        assertEquals(0, symbolMap.size());
    }
    
    @Test
    public void testSize_2() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", "value1");
        
        assertEquals(1, symbolMap.size());
    }
    
    @Test
    public void testSize_3() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        assertEquals(entryCount, symbolMap.size());
    }
    
    @Test
    public void testSize_4() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap.addSymbol("symbol" + i, "value" + i);
            }
            else {
                symbolMap.addSymbol("symbol" + i, null);
            }
        }
        
        assertEquals(entryCount, symbolMap.size());
    }
    
    @Test
    public void testSize_5() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            symbolMap.removeSymbol("symbol" + i);
        }
        
        assertEquals(2 * 150, symbolMap.size());
    }
    
    @Test
    public void testIsEmpty_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        assertTrue(symbolMap.isEmpty());
    }
    
    @Test
    public void testIsEmpty_2() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", "value1");
        
        assertFalse(symbolMap.isEmpty());
    }
    
    @Test
    public void testIsEmpty_3() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        assertFalse(symbolMap.isEmpty());
    }
    
    @Test
    public void testIsEmpty_4() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap.addSymbol("symbol" + i, "value" + i);
            }
            else {
                symbolMap.addSymbol("symbol" + i, null);
            }
        }
        
        assertFalse(symbolMap.isEmpty());
    }
    
    @Test
    public void testIsEmpty_5() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            symbolMap.removeSymbol("symbol" + i);
        }
        
        assertFalse(symbolMap.isEmpty());
    }
    
    @Test
    public void testContainsSymbol_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        assertFalse(symbolMap.containsSymbol("symbol"));
    }
    
    @Test
    public void testContainsSymbol_2() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", "value1");
        
        assertTrue(symbolMap.containsSymbol("symbol1"));
    }
    
    @Test
    public void testContainsSymbol_3() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 0; i < entryCount; i++) {
            assertTrue(symbolMap.containsSymbol("symbol" + i));
        }
    }
    
    @Test
    public void testContainsSymbol_4() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap.addSymbol("symbol" + i, "value" + i);
            }
            else {
                symbolMap.addSymbol("symbol" + i, null);
            }
        }
        
        for (int i = 0; i < entryCount; i++) {
            assertTrue(symbolMap.containsSymbol("symbol" + i));
        }
    }
    
    @Test
    public void testContainsSymbol_5() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            symbolMap.removeSymbol("symbol" + i);
        }
        
        for (int i = 0; i < 150; i++) {
            assertTrue(symbolMap.containsSymbol("symbol" + i));
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            assertFalse(symbolMap.containsSymbol("symbol" + i));
        }
        
        for (int i = (entryCount - 150); i < entryCount; i++) {
            assertTrue(symbolMap.containsSymbol("symbol" + i));
        }
    }
    
    @Test(expected = java.lang.IllegalArgumentException.class)
    public void testContainsSymbol_6() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.containsSymbol(null);
    }    

    @Test
    public void testGetValue_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        assertNull(symbolMap.getValue("symbol"));
    }
    
    @Test
    public void testGetValue_2() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", "value1");
        
        assertNotNull(symbolMap.getValue("symbol1"));
    }
    
    @Test
    public void testGetValue_3() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", null);
        
        assertNull(symbolMap.getValue("symbol1"));
    }
    
    @Test
    public void testGetValue_4() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 0; i < entryCount; i++) {
            assertNotNull(symbolMap.getValue("symbol" + i));
        }
    }
    
    @Test
    public void testGetValue_5() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap.addSymbol("symbol" + i, "value" + i);
            }
            else {
                symbolMap.addSymbol("symbol" + i, null);
            }
        }
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                assertNotNull(symbolMap.getValue("symbol" + i));
            }
            else {
                assertNull(symbolMap.getValue("symbol" + i));
            }
        }
    }
    
    @Test
    public void testGetValue_6() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            symbolMap.removeSymbol("symbol" + i);
        }
        
        for (int i = 0; i < 150; i++) {
            assertNotNull(symbolMap.getValue("symbol" + i));
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            assertNull(symbolMap.getValue("symbol" + i));
        }
        
        for (int i = (entryCount - 150); i < entryCount; i++) {
            assertNotNull(symbolMap.getValue("symbol" + i));
        }
    }
    
    @Test(expected = java.lang.IllegalArgumentException.class)
    public void testGetValue_7() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.getValue(null);
    }    

    @Test
    public void testRemoveSymbol_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        assertNull(symbolMap.removeSymbol("symbol"));
    }
    
    @Test
    public void testRemoveSymbol_2() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", "value1");
        
        assertNotNull(symbolMap.removeSymbol("symbol1"));
        assertEquals(0, symbolMap.size());
    }
    
    @Test
    public void testRemoveSymbol_3() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", null);
        
        assertNull(symbolMap.removeSymbol("symbol1"));
        assertEquals(0, symbolMap.size());
    }
    
    @Test
    public void testRemoveSymbol_4() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 0; i < entryCount; i++) {
            assertNotNull(symbolMap.removeSymbol("symbol" + i));
        }
        
        assertEquals(0, symbolMap.size());
    }
    
    @Test
    public void testRemoveSymbol_5() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap.addSymbol("symbol" + i, "value" + i);
            }
            else {
                symbolMap.addSymbol("symbol" + i, null);
            }
        }
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                assertNotNull(symbolMap.removeSymbol("symbol" + i));
            }
            else {
                assertNull(symbolMap.removeSymbol("symbol" + i));
            }
        }
        
        assertEquals(0, symbolMap.size());
    }
    
    @Test
    public void testRemoveSymbol_6() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            symbolMap.removeSymbol("symbol" + i);
        }
        
        for (int i = 0; i < 150; i++) {
            assertNotNull(symbolMap.removeSymbol("symbol" + i));
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            assertNull(symbolMap.removeSymbol("symbol" + i));
        }
        
        for (int i = (entryCount - 150); i < entryCount; i++) {
            assertNotNull(symbolMap.removeSymbol("symbol" + i));
        }
        
        assertEquals(0, symbolMap.size());
    }
    
    @Test(expected = java.lang.IllegalArgumentException.class)
    public void testRemoveSymbol_7() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.removeSymbol(null);
    }    
   
    @Test
    public void testGetSymbols_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        assertNotNull(symbolMap.getSymbols());
        assertEquals(0, symbolMap.getSymbols().size());
    }
    
    @Test
    public void testGetSymbols_2() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", "value1");
        
        assertNotNull(symbolMap.getSymbols());
        assertEquals(1, symbolMap.getSymbols().size());
        assertEquals("symbol1", symbolMap.getSymbols().iterator().next());
    }
    
    @Test
    public void testGetSymbols_3() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", null);
        
        assertNotNull(symbolMap.getSymbols());
        assertEquals(1, symbolMap.getSymbols().size());
        assertEquals("symbol1", symbolMap.getSymbols().iterator().next());
    }
    
    @Test
    public void testGetSymbols_4() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        assertNotNull(symbolMap.getSymbols());
        assertEquals(entryCount, symbolMap.getSymbols().size());
        
        Iterator<String> iterator = symbolMap.getSymbols().iterator();
        for (int i = 0; i < entryCount; i++) {
            assertTrue(iterator.hasNext());
            assertEquals("symbol" + i, iterator.next());
        }
        
        assertFalse(iterator.hasNext());
    }
    
    @Test
    public void testGetSymbols_5() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap.addSymbol("symbol" + i, "value" + i);
            }
            else {
                symbolMap.addSymbol("symbol" + i, null);
            }
        }
        
        assertNotNull(symbolMap.getSymbols());
        assertEquals(entryCount, symbolMap.getSymbols().size());
        
        Iterator<String> iterator = symbolMap.getSymbols().iterator();
        for (int i = 0; i < entryCount; i++) {
            assertTrue(iterator.hasNext());
            assertEquals("symbol" + i, iterator.next());
        }
        
        assertFalse(iterator.hasNext());
    }
    
    @Test
    public void testGetSymbols_6() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            symbolMap.removeSymbol("symbol" + i);
        }
        
        assertNotNull(symbolMap.getSymbols());
        assertEquals(2 * 150, symbolMap.getSymbols().size());
        
        Iterator<String> iterator = symbolMap.getSymbols().iterator();
        for (int i = 0; i < 2 * 150; i++) {
            assertTrue(iterator.hasNext());
            
            if (i < 150) {
                assertEquals("symbol" + i, iterator.next());
            }
            else if (i >= 150){
                assertEquals("symbol" + (entryCount - 300 + i), iterator.next());
            }
        }
        
        assertFalse(iterator.hasNext());
    }
    
    @Test
    public void testGetValues_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        assertNotNull(symbolMap.getValues());
        assertEquals(0, symbolMap.getValues().size());
    }
    
    @Test
    public void testGetValues_2() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", "value1");
        
        assertNotNull(symbolMap.getValues());
        assertEquals(1, symbolMap.getValues().size());
        assertEquals("value1", symbolMap.getValues().iterator().next());
    }
    
    @Test
    public void testGetValues_3() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", null);
        
        assertNotNull(symbolMap.getValues());
        assertEquals(1, symbolMap.getValues().size());
        assertNull(symbolMap.getValues().iterator().next());
    }
    
    @Test
    public void testGetValues_4() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        assertNotNull(symbolMap.getValues());
        assertEquals(entryCount, symbolMap.getValues().size());
        
        Iterator<String> iterator = symbolMap.getValues().iterator();
        for (int i = 0; i < entryCount; i++) {
            assertTrue(iterator.hasNext());
            assertEquals("value" + i, iterator.next());
        }
        
        assertFalse(iterator.hasNext());
    }
    
    @Test
    public void testGetValues_5() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap.addSymbol("symbol" + i, "value" + i);
            }
            else {
                symbolMap.addSymbol("symbol" + i, null);
            }
        }
        
        assertNotNull(symbolMap.getValues());
        assertEquals(entryCount, symbolMap.getValues().size());
        
        Iterator<String> iterator = symbolMap.getValues().iterator();
        for (int i = 0; i < entryCount; i++) {
            assertTrue(iterator.hasNext());
            if (i % 7 == 0) {
                assertEquals("value" + i, iterator.next());
            }
            else {
                assertNull(iterator.next());
            }
        }
        
        assertFalse(iterator.hasNext());
    }
    
    @Test
    public void testGetValues_6() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            symbolMap.removeSymbol("symbol" + i);
        }
        
        assertNotNull(symbolMap.getValues());
        assertEquals(2 * 150, symbolMap.getValues().size());
        
        Iterator<String> iterator = symbolMap.getValues().iterator();
        for (int i = 0; i < 2 * 150; i++) {
            assertTrue(iterator.hasNext());
            
            if (i < 150) {
                assertEquals("value" + i, iterator.next());
            }
            else if (i >= 150){
                assertEquals("value" + (entryCount - 300 + i), iterator.next());
            }
        }
        
        assertFalse(iterator.hasNext());
    }
    
    @Test
    public void testClear_1() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        symbolMap.clear();
        assertEquals(0, symbolMap.getValues().size());
    }
    
    @Test
    public void testClear_2() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", "value1");
        
        symbolMap.clear();
        assertEquals(0, symbolMap.getValues().size());
    }
    
    @Test
    public void testClear_3() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap.addSymbol("symbol1", null);
        
        symbolMap.clear();
        assertEquals(0, symbolMap.getValues().size());
    }
    
    @Test
    public void testClear_4() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        symbolMap.clear();
        assertEquals(0, symbolMap.getValues().size());
    }
    
    @Test
    public void testClear_5() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap.addSymbol("symbol" + i, "value" + i);
            }
            else {
                symbolMap.addSymbol("symbol" + i, null);
            }
        }
        
        symbolMap.clear();
        assertEquals(0, symbolMap.getValues().size());
    }
    
    @Test
    public void testClear_6() {
        SymbolMap<String, String> symbolMap =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 2000;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap.addSymbol("symbol" + i, "value" + i);
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            symbolMap.removeSymbol("symbol" + i);
        }
        
        symbolMap.clear();
        assertEquals(0, symbolMap.getValues().size());
    }
    
    @Test
    public void testEquals_1() {
        SymbolMap<String, String> symbolMap1 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        SymbolMap<String, String> symbolMap2 =
                new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        assertTrue(symbolMap1.equals(symbolMap2));
    }
    
    @Test
    public void testEquals_2() {
        SymbolMap<String, String> symbolMap1 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        SymbolMap<String, String> symbolMap2 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap1.addSymbol("symbol1", "value1");
        
        assertFalse(symbolMap1.equals(symbolMap2));
    }
    
    @Test
    public void testEquals_3() {
        SymbolMap<String, String> symbolMap1 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        SymbolMap<String, String> symbolMap2 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
            
        symbolMap1.addSymbol("symbol1", "value1");
        symbolMap2.addSymbol("symbol1", "value1");
        
        assertTrue(symbolMap1.equals(symbolMap2));
    }

    @Test
    public void testEquals_4() {
        SymbolMap<String, String> symbolMap1 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
                
        SymbolMap<String, String> symbolMap2 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 200;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap1.addSymbol("symbol" + i, "value" + i);
            assertFalse(symbolMap1.equals(symbolMap2));
            symbolMap2.addSymbol("symbol" + i, "value" + i);
            assertTrue(symbolMap1.equals(symbolMap2));
        }
    }
    
    @Test
    public void testEquals_5() {
        SymbolMap<String, String> symbolMap1 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
               
        SymbolMap<String, String> symbolMap2 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 200;
        
        for (int i = 0; i < entryCount; i++) {
            if (i % 7 == 0) {
                symbolMap1.addSymbol("symbol" + i, "value" + i);
                assertFalse(symbolMap1.equals(symbolMap2));
                symbolMap2.addSymbol("symbol" + i, "value" + i);
                assertTrue(symbolMap1.equals(symbolMap2));
            }
            else {
                symbolMap1.addSymbol("symbol" + i, null);
                assertFalse(symbolMap1.equals(symbolMap2));
                symbolMap2.addSymbol("symbol" + i, null);
                assertTrue(symbolMap1.equals(symbolMap2));
            }
        }
    }
    
    @Test
    public void testEquals_6() {
        SymbolMap<String, String> symbolMap1 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
                
        SymbolMap<String, String> symbolMap2 =
            new SymbolMap<String, String>(new DefaultSymbolComparator<String>());
        
        int entryCount = 200;
        
        for (int i = 0; i < entryCount; i++) {
            symbolMap1.addSymbol("symbol" + i, "value" + i);
            assertFalse(symbolMap1.equals(symbolMap2));
            symbolMap2.addSymbol("symbol" + i, "value" + i);
            assertTrue(symbolMap1.equals(symbolMap2));
        }
        
        for (int i = 150; i < (entryCount - 150); i++) {
            symbolMap1.removeSymbol("symbol" + i);
            assertFalse(symbolMap1.equals(symbolMap2));
            symbolMap2.removeSymbol("symbol" + i);
            assertTrue(symbolMap1.equals(symbolMap2));
        }
    }
    
    private void assertSymbolMapEntries(SymbolMap<String, String> symbolMap,
                                        String[]                  symbols,
                                        String[]                  values)
    {
        assertEquals(symbols.length, symbolMap.size());
        assertEquals(symbols.length, symbolMap.getSymbols().size());
        assertEquals(values.length, symbolMap.getValues().size());
        
        for (int i = 0; i < symbols.length; i++) {
            assertTrue(symbolMap.containsSymbol(symbols[i]));
            assertEquals(values[i], symbolMap.getValue(symbols[i]));
            assertTrue(symbolMap.getSymbols().contains(symbols[i]));
            assertTrue(symbolMap.getValues().contains(values[i]));            
        }
    }
}
