//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usageprofiles;

import java.io.Serializable;

/**
 * <p>
 * This interface can be used for implementing comparison strategies for symbols.
 * </p>
 * 
 * @author Patrick Harms
 */
public interface SymbolComparator<T> extends Serializable {
    
    /**
     * <p>
     * compares two symbols and returns true, if the concrete comparison strategy sees both
     * symbols as equal. The method must be commutative, i.e.,
     * <code>equals(symbol1, symbol2) == equals(symbol2, symbol1)</code>.
     * </p>
     *
     * @param symbol1 the first symbol to be compared
     * @param symbol2 the second symbol to be compared
     * 
     * @return true if the comparison strategy sees both symbols as equal, false else.
     */
    public boolean equals(T symbol1, T symbol2);
    
    /**
     * <p>
     * returns a search order for buckets. This method can be used for optimizing a search for
     * equal symbols. The client of this class can store symbols in buckets of similar symbols.
     * Those buckets get an id denoted by an integer. The most appropriate bucket for a symbol is
     * the first element in the array returned by this method. The symbol should therefore be put
     * into that bucket.
     * </p>
     * <p>
     * In case a search for an equal symbol shall be performed at the client side, the client
     * checks the available buckets in the order given as return value of this method. All other
     * buckets are searched afterwards. In this scenario it is ensured, that the equal symbol is
     * found as soon as possible as the search always starts in the most appropriate bucket.
     * However, the other buckets are searched, as well, if no equal symbol is found. Therefore,
     * in the worst case, all buckets are searched. In the optimal case, the equal symbol is found
     * in the first bucket.
     * </p>
     * <p>
     * The returned array should contain different integers in each field. This allows a most
     * efficient search. If an integer is repeated, it is up to the clien, if this is ignored.
     * </p>
     *
     * @param symbol the symbol for which the bucket order is to be returned
     * 
     * @return a search order for buckets as described
     * 
     * @see SymbolMap
     */
    public int[] getBucketSearchOrder(T symbol);
}
