//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.eventcore.guimodel;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * <p>
 * This class is a dummy GUI element to represent groups of GUI elements. A group of GUI elements
 * can be integrated in any GUI model using the method
 * {@link GUIModel#groupGUIElements(java.util.List, String)}. A group has the same behavior as
 * any other parent GUI element.
 * </p>
 * 
 * @author Patrick Harms
 */
public class GUIElementGroup extends AbstractDefaultGUIElement {

    /**  */
    private static final long serialVersionUID = 1L;
    
    /**
     * the list of grouped GUIElements
     */
    private List<IGUIElement> groupedGUIElements = new LinkedList<IGUIElement>();

    /**
     * <p>
     * instantiates a GUI element group with a name and its optional parent GUI element
     * </p>
     *
     * @param groupName the name of the GUI element group
     * @param parent    the optional parent GUI element of the group
     */
    public GUIElementGroup(String groupName, IGUIElement parent) {
        super(new GroupSpecification(groupName), parent);
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#updateSpecification(IGUIElementSpec)
     */
    @Override
    public final void updateSpecification(IGUIElementSpec furtherSpec) {
        // do nothing
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.IEventTarget#getPlatform()
     */
    @Override
    public String getPlatform() {
        return "none";
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.IEventTarget#getStringIdentifier()
     */
    @Override
    public String getStringIdentifier() {
        return ((GroupSpecification) super.getSpecification()).name;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return getStringIdentifier();
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param guiElement
     */
    public void addToGroup(IGUIElement guiElement) {
        this.groupedGUIElements.add(guiElement);
    }
    
    /**
     * 
     */
    public List<IGUIElement> getGroupedElements() {
        return Collections.unmodifiableList(groupedGUIElements);
    }
    
    /**
     * <p>
     * internally required GUI element specification for a GUI element group. This is just a wrapper
     * for a name of a GUI element group
     * </p>
     * 
     * @author Patrick Harms
     */
    private static class GroupSpecification implements IGUIElementSpec {
        
        /**  */
        private static final long serialVersionUID = 1L;
        /**
         * the name of the GUI element group represented by this specification
         */
        private String name;

        /**
         * <p>
         * instantiates the group specification with the given name. Two group specifications
         * are only similar, if their names match.
         * </p>
         *
         * @param name the name of the group
         */
        private GroupSpecification(String name) {
            super();
            this.name = name;
        }

        /* (non-Javadoc)
         * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec#getType()
         */
        @Override
        public String getType() {
            return "GUI element group";
        }

        /* (non-Javadoc)
         * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec#getTypeHierarchy()
         */
        @Override
        public String[] getTypeHierarchy() {
            return new String[] { getType() };
        }

        /* (non-Javadoc)
         * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec#getSimilarity(IGUIElementSpec)
         */
        @Override
        public boolean getSimilarity(IGUIElementSpec other) {
            return
                (other instanceof GroupSpecification) &&
                name.equals(((GroupSpecification) other).name);
        }

    }

}
