//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.treeifc;

import java.util.List;

/**
 * <p>
 * a task instance represents the execution of a specific task within a user session. A task
 * instance is always related to the task that was executed. A task instance may have children
 * but only if its related task allows to have children. For example, if the represented task is
 * a sequence then task instance has children and these are instances of the tasks being the
 * children of the related sequence.
 * </p>
 * <p>
 * An instance of a sequence has the same number of children as the related sequence. An instance
 * of a selection has only one child which is an instance of exactly one variant contained in the
 * related selection. An instance of an iteration has zero or more instances of the same task
 * as children where the task is the child of the related iteration. An instance of an optional
 * has zero or one child where the task related to child is the child of the optional. A task
 * instance related to an event task does not have children.
 * </p>
 * 
 * @author Patrick Harms
 */
public interface ITaskInstance extends ITaskInstanceList {

    /**
     * <p>
     * returns the children of the task instance if any. See class description for how many
     * children can be expected. May return null.
     * </p>
     * 
     * @return as described
     */
    public List<ITaskInstance> getChildren();

    /**
     * <p>
     * returns the task related to the instance.
     * </p>
     * 
     * @return as described
     */
    public ITask getTask();

    /**
     * <p>
     * compares an instance to another one. Returns true if both instances are the same, i.e. not
     * only the related task is equal but also all children and further characteristics of the
     * task instance  
     * </p>
     * 
     * @param taskInstance the instance to compare to
     * 
     * @return as described
     */
    public boolean equals(ITaskInstance taskInstance);

    /**
     * <p>
     * returns a hash code for the task instance to be able to put it into hash maps
     * </p>
     * 
     * @return as described
     */
    public int hashCode();

    /**
     * <p>
     * clones a task instance by creating exact clones of each contained child instance as well
     * as the related task. Furthermore, all other non transient information of the task
     * instance must be cloned.
     * </p>
     * 
     * @return a clone of the task instance
     */
    public ITaskInstance clone();

}
