//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import java.util.ArrayList;
import java.util.List;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;

/**
 * <p>
 * The task equality rule manager is capable of comparing tasks based on its internal list
 * of comparison rules. These rules are asked for comparing the two provided tasks. If a rule
 * returns a task equality other than null, this equality is returned. Otherwise the next rule
 * is asked.
 * </p>
 * 
 * @version $Revision: $ $Date: 19.02.2012$
 * @author 2012, last modified by $Author: patrick$
 */
public class TaskEqualityRuleManager {
    
    /**
     * <p>
     * the singleton instance of this class
     * </p>
     */
    private static final TaskEqualityRuleManager instance = new TaskEqualityRuleManager();

    /**
     * <p>
     * the rules that can be used for comparing tasks
     * </p>
     */
    private List<TaskComparisonRule> mRuleIndex = null;

    /**
     * <p>
     * initializes the task equality rule manager by filling the internal list of comparison rules.
     * </p>
     */
    private TaskEqualityRuleManager() {
        mRuleIndex = new ArrayList<TaskComparisonRule>();
        mRuleIndex.add(new TaskIdentityRule());
        mRuleIndex.add(new GUIEventTaskComparisonRule());
        mRuleIndex.add(new EventTaskComparisonRule());
        mRuleIndex.add(new IterationComparisonRule());
        mRuleIndex.add(new SequenceComparisonRule());
        mRuleIndex.add(new SelectionComparisonRule());
        mRuleIndex.add(new TaskAndIterationComparisonRule());
        mRuleIndex.add(new TaskAndSelectionComparisonRule());
    }


    /**
     * <p>
     * returns the singleton instance of this class
     * </p>
     * 
     * @return as described
     */
    public static TaskEqualityRuleManager getInstance() {
        return instance;
    }

    /**
     * <p>
     * this method performs a comparison of the two provided tasks. It iterates its internal
     * comparison rules. If the first rule returns a task equality other than null,
     * this equality is returned. Otherwise the next rule is tried. If no rule returns an equality
     * <code>NodeEquality.UNEQUAL</code> is returned.
     * </p>
     * 
     * @param task1 the first task to be compared
     * @param task2 the second task to be compared
     * 
     * @return as described
     * 
     * @throws IllegalStateException in the case, the {@link #init()} method was not called on the
     *                               manager before a call to this method.
     */
    public TaskEquality compare(ITask task1, ITask task2)
        throws IllegalStateException
    {
        if (mRuleIndex == null) {
            throw new IllegalStateException("not initialized");
        }
        
        // LOG.info("checking for equality of " + task1 + " and " + task2);
        TaskEquality taskEquality = null;

        for (TaskComparisonRule rule : mRuleIndex) {
            if (rule.isApplicable(task1, task2)) {
                taskEquality = rule.compare(task1, task2);
                if (taskEquality != null) {
                    // LOG.warning("used rule " + rule + " for equality check");
                    return taskEquality;
                }
            }
        }

        // LOG.warning("no rule could be applied --> handling tasks as unequal");

        return TaskEquality.UNEQUAL;
    }

    /**
     * <p>
     * this method two tasks with respect to the fiven equality level and returns true, if this
     * level is given.
     * </p>
     * 
     * @param task1         the first task to be compared
     * @param task2         the second task to be compared
     * @param equalityLevel the level of equality to be checked for
     * 
     * @return as described
     * 
     * @throws IllegalStateException in the case, the {@link #init()} method was not called on the
     *                               manager before a call to this method.
     */
    public boolean areAtLeastEqual(ITask task1, ITask task2, TaskEquality equalityLevel) {
        if (equalityLevel == null) {
            throw new IllegalArgumentException("required equality level must not be null");
        }
        
        switch (equalityLevel) {
            case IDENTICAL:
                return areIdentical(task1, task2);
            case LEXICALLY_EQUAL:
                return areLexicallyEqual(task1, task2);
            case SYNTACTICALLY_EQUAL:
                return areSyntacticallyEqual(task1, task2);
            case SEMANTICALLY_EQUAL:
                return areSemanticallyEqual(task1, task2);
            case UNEQUAL:
                return !areSemanticallyEqual(task1, task2);
            default:
                throw new IllegalArgumentException("unknown required equality: " + equalityLevel);
        }
    }

    /**
     * <p>
     * this method checks if the two given tasks are identical. For this, it iterates its internal
     * comparison rules. If the first rule returns true, than this method returns true as well.
     * If no rule returns true, this method returns false.
     * </p>
     * 
     * @param task1 the first task to be compared
     * @param task2 the second task to be compared
     * 
     * @return as described
     * 
     * @throws IllegalStateException in the case, the {@link #init()} method was not called on the
     *                               manager before a call to this method.
     */
    public boolean areIdentical(ITask task1, ITask task2) {
        if (mRuleIndex == null) {
            throw new IllegalStateException("not initialized");
        }
        
        for (TaskComparisonRule rule : mRuleIndex) {
            if (rule.isApplicable(task1, task2) && rule.areLexicallyEqual(task1, task2)) {
                 return true;
            }
        }

        return false;
    }

    /**
     * <p>
     * this method checks if the two given tasks are lexically equal. For this, it iterates its
     * internal comparison rules. If the first rule returns true, than this method returns true
     * as well. If no rule returns true, this method returns false.
     * </p>
     * 
     * @param task1 the first task to be compared
     * @param task2 the second task to be compared
     * 
     * @return as described
     * 
     * @throws IllegalStateException in the case, the {@link #init()} method was not called on the
     *                               manager before a call to this method.
     */
    public boolean areLexicallyEqual(ITask task1, ITask task2) {
        if (mRuleIndex == null) {
            throw new IllegalStateException("not initialized");
        }
        
        for (TaskComparisonRule rule : mRuleIndex) {
            if (rule.isApplicable(task1, task2) && rule.areLexicallyEqual(task1, task2)) {
                 return true;
            }
        }

        return false;
    }

    /**
     * <p>
     * this method checks if the two given tasks are syntactically equal. For this, it iterates its
     * internal comparison rules. If the first rule returns true, than this method returns true
     * as well. If no rule returns true, this method returns false.
     * </p>
     * 
     * @param task1 the first task to be compared
     * @param task2 the second task to be compared
     * 
     * @return as described
     * 
     * @throws IllegalStateException in the case, the {@link #init()} method was not called on the
     *                               manager before a call to this method.
     */
    public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
        if (mRuleIndex == null) {
            throw new IllegalStateException("not initialized");
        }
        
        for (TaskComparisonRule rule : mRuleIndex) {
            if (rule.isApplicable(task1, task2) && rule.areSyntacticallyEqual(task1, task2)) {
                 return true;
            }
        }

        return false;
    }

    /**
     * <p>
     * this method checks if the two given tasks are semantically equal. For this, it iterates its
     * internal comparison rules. If the first rule returns true, than this method returns true
     * as well. If no rule returns true, this method returns false.
     * </p>
     * 
     * @param task1 the first task to be compared
     * @param task2 the second task to be compared
     * 
     * @return as described
     * 
     * @throws IllegalStateException in the case, the {@link #init()} method was not called on the
     *                               manager before a call to this method.
     */
    public boolean areSemanticallyEqual(ITask task1, ITask task2) {
        if (mRuleIndex == null) {
            throw new IllegalStateException("not initialized");
        }
        
        for (TaskComparisonRule rule : mRuleIndex) {
            if (rule.isApplicable(task1, task2) && rule.areSemanticallyEqual(task1, task2)) {
                 return true;
            }
        }

        return false;
    }

}
