//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules.metrics;

import static de.ugoe.cs.autoquest.usability.taskmodel.filter.types.EventTypeFilter.TEXT_INPUT;
import static de.ugoe.cs.autoquest.usability.util.TextInputUtil.aggregateEnteredTextFromTextInputs;
import static de.ugoe.cs.autoquest.usability.util.TextInputUtil.characterIsNoLetterOrDigitPredicate;

import java.util.List;

import com.google.common.base.CharMatcher;
import com.google.common.base.Optional;
import com.google.common.collect.Multiset;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.usability.EvaluationMethodCaller;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescriptionResolver;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescription;
import de.ugoe.cs.autoquest.usability.rules.UsabilityMetric;
import de.ugoe.cs.autoquest.usability.rules.UsabilityRule;
import de.ugoe.cs.autoquest.usability.taskmodel.filter.FilterResult;
import de.ugoe.cs.autoquest.usability.taskmodel.filter.IterativeDFSFilterStrategy;
import de.ugoe.cs.autoquest.usability.taskmodel.filter.types.TaskModelFilter;

/**
 * <p>
 * Metric, which calculates the ratio ratio between non letter or digit characters and all entered
 * characters.
 * </p>
 * 
 * @author Alexander Deicke
 */
public class NoLetterOrDigitRatioMetric extends UsabilityRule implements UsabilityMetric {

    /**
     * <p>
     * Constructor. Creates a new {@code NoLetterOrDigitRatioMetric} for a given task model.
     * </p>
     * 
     * @param taskModel
     */
    public NoLetterOrDigitRatioMetric(ITaskModel taskModel) {
        super(taskModel);
        this.name = "NoLetterOrDigitRatio";
        this.defect =
            new UsabilityProblemDescriptionResolver().descriptionFor(this.getClass()
                .getSimpleName());
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.usability.rules.UsabilityRule#check()
     */
    @Override
    public Optional<UsabilityProblemDescription> calculate() {
        FilterResult textInputEvents = extractNodesFromTaskTree();
        float evaluationMetric = calculateEvaluationMetric(textInputEvents.tasksMatchedFilter());
        return this.defect.isPresent(evaluationMetric);
    }

    /**
     * 
     * <p>
     * Filters all text input events from task model.
     * </p>
     * 
     * @return {@code FilterResult}
     */
    private FilterResult extractNodesFromTaskTree() {
        return new TaskModelFilter(new IterativeDFSFilterStrategy()).filterByEventType(TEXT_INPUT)
            .from(this.taskModel);
    }

    /**
     * 
     * <p>
     * Calculates the metric.
     * </p>
     * 
     * @param textInputEvents
     *            all text input events
     * @return ratio between non letter or digit characters and all entered characters
     */
    private float calculateEvaluationMetric(List<ITask> textInputEvents) {
        Multiset<String> enteredTextFragments = aggregateEnteredTextFromTextInputs(textInputEvents);
        int allCharactersCount = 0;
        int noLetterOrDigitCount = 0;
        for (String textFragment : enteredTextFragments.elementSet()) {
            int occurencesOfTextFragment = enteredTextFragments.count(textFragment);
            allCharactersCount += CharMatcher.ANY.countIn(textFragment) * occurencesOfTextFragment;
            noLetterOrDigitCount +=
                CharMatcher.forPredicate(characterIsNoLetterOrDigitPredicate())
                    .countIn(textFragment) * occurencesOfTextFragment;
        }
        return allCharactersCount != 0 ? (float) noLetterOrDigitCount / (float) allCharactersCount
            : 0;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * de.ugoe.cs.autoquest.usability.rules.UsabilityRule#callEvaluationMetho(de.ugoe.cs.autoquest
     * .usability.EvaluationMethodCaller)
     */
    @Override
    public Optional<UsabilityProblemDescription> callEvaluationMethod(EvaluationMethodCaller evaluationMethodCaller)
    {
        return evaluationMethodCaller.check(this);
    }

}
