//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.manager;

import java.util.logging.Level;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.IEventTarget;
import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.tasktrees.TaskTreeChecker;
import de.ugoe.cs.autoquest.tasktrees.manager.ComponentManager;
import de.ugoe.cs.autoquest.tasktrees.manager.TaskTreeManager;
import de.ugoe.cs.autoquest.test.DummyGUIElement;
import de.ugoe.cs.autoquest.test.DummyInteraction;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.TextConsole;

/**
 * TODO comment
 * 
 * @version $Revision: $ $Date: $
 * @author 2011, last modified by $Author: $
 */
public class TaskTreeManagerTest {
    
    /** */
    TaskTreeManager manager;

    /**
     *
     */
    @Before
    public void setUp() {
        Console.reset();
        new TextConsole(Level.FINEST);
        manager = new TaskTreeManager();
    }

    /**
     *
     */
    @After
    public void tearDown() {
        manager = null;
        ComponentManager.clearInstance();
    }

    /**
     *
     */
    @Test
    public void testOneEventOnOneElement() {
        simulateEvent(new DummyInteraction("bla", 1), new DummyGUIElement("elem1"));
        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Event bla {}" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    @Test
    public void testManyEventsOnOneElement() {
        IEventTarget eventTarget = new DummyGUIElement("elem1");
        simulateEvent(new DummyInteraction("bla", 1), eventTarget);
        simulateEvent(new DummyInteraction("bli", 1), eventTarget);
        simulateEvent(new DummyInteraction("blo", 1), eventTarget);
        simulateEvent(new DummyInteraction("blu", 1), eventTarget);
        simulateEvent(new DummyInteraction("ble", 1), eventTarget);

        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Event bla {}" +
             "  Event bli {}" +
             "  Event blo {}" +
             "  Event blu {}" +
             "  Event ble {}" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    @Test
    public void testOneEventOnManyElements() {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        IEventTarget eventTarget3 = new DummyGUIElement("elem3");
        IEventTarget eventTarget4 = new DummyGUIElement("elem4");
        IEventTarget eventTarget5 = new DummyGUIElement("elem5");
        IEventTarget eventTarget6 = new DummyGUIElement("elem6");
        simulateEvent(new DummyInteraction("bla", 1), eventTarget1);
        simulateEvent(new DummyInteraction("bli", 1), eventTarget2);
        simulateEvent(new DummyInteraction("blo", 1), eventTarget3);
        simulateEvent(new DummyInteraction("blu", 1), eventTarget4);
        simulateEvent(new DummyInteraction("ble", 1), eventTarget5);
        simulateEvent(new DummyInteraction("blum", 1), eventTarget6);

        new TaskTreeChecker(true).assertTaskInstanceList
            ("UserSession session {" +
             "  Event bla {}" +
             "  Event bli {}" +
             "  Event blo {}" +
             "  Event blu {}" +
             "  Event ble {}" +
             "  Event blum {}" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    @Test
    public void testManyEventsOnManyElements() {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventTarget eventTarget2 = new DummyGUIElement("elem2");
        IEventTarget eventTarget3 = new DummyGUIElement("elem3");
        IEventTarget eventTarget4 = new DummyGUIElement("elem4");
        IEventTarget eventTarget5 = new DummyGUIElement("elem5");
        IEventTarget eventTarget6 = new DummyGUIElement("elem6");
        simulateEvent(new DummyInteraction("bla", 1), eventTarget1);
        simulateEvent(new DummyInteraction("bli", 1), eventTarget1);
        simulateEvent(new DummyInteraction("blup", 1), eventTarget1);
        simulateEvent(new DummyInteraction("ble", 1), eventTarget2);
        simulateEvent(new DummyInteraction("blo", 1), eventTarget2);
        simulateEvent(new DummyInteraction("blau", 1), eventTarget3);
        simulateEvent(new DummyInteraction("blass", 1), eventTarget4);
        simulateEvent(new DummyInteraction("blum", 1), eventTarget4);
        simulateEvent(new DummyInteraction("blim", 1), eventTarget4);
        simulateEvent(new DummyInteraction("blom", 1), eventTarget4);
        simulateEvent(new DummyInteraction("blam", 1), eventTarget5);
        simulateEvent(new DummyInteraction("blip", 1), eventTarget6);
        simulateEvent(new DummyInteraction("blap", 1), eventTarget6);
        simulateEvent(new DummyInteraction("blep", 1), eventTarget6);
        simulateEvent(new DummyInteraction("blop", 1), eventTarget6);

        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Event bla {}" +
             "  Event bli {}" +
             "  Event blup {}" +
             "  Event ble {}" +
             "  Event blo {}" +
             "  Event blau {}" +
             "  Event blass {}" +
             "  Event blum {}" +
             "  Event blim {}" +
             "  Event blom {}" +
             "  Event blam {}" +
             "  Event blip {}" +
             "  Event blap {}" +
             "  Event blep {}" +
             "  Event blop {}" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    @Test
    public void testEventIterationDetection_01() throws Exception {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventType event1 = new DummyInteraction("bla", 1);
        simulateEvent(event1, eventTarget1);
        simulateEvent(event1, eventTarget1);
        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Iteration iteration1 {" +
             "    Event bla {}" +
             "    Event bla {}" +
             "  }" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    @Test
    public void testEventIterationDetection_02() throws Exception {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventType event1 = new DummyInteraction("bla", 1);
        for (int i = 0; i < 10; i++) {
            simulateEvent(event1, eventTarget1);
        }

        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Iteration iteration1 {" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "  }" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    @Test
    public void testEventIterationDetection_03() throws Exception {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventType event1 = new DummyInteraction("bla", 1);

        // now test with preceding and trailing other interactions
        IEventType event2 = new DummyInteraction("bli", 1);
        IEventType event3 = new DummyInteraction("blup", 1);
        IEventType event4 = new DummyInteraction("ble", 1);
        IEventType event5 = new DummyInteraction("blo", 1);

        simulateEvent(event2, eventTarget1);
        simulateEvent(event3, eventTarget1);
        for (int i = 0; i < 10; i++) {
            simulateEvent(event1, eventTarget1);
        }
        simulateEvent(event4, eventTarget1);
        simulateEvent(event5, eventTarget1);

        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Event bli {}" +
             "  Event blup {}" +
             "  Iteration iteration2 {" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "    Event bla {}" +
             "  }" +
             "  Event ble {}" +
             "  Event blo {}" +
             "}", manager.getTaskModel().getUserSessions().get(0));

    }

    /**
     *
     */
    @Test
    public void testEventIterationDetection_04() throws Exception {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventType event1 = new DummyInteraction("bla", 1);
        IEventType event2 = new DummyInteraction("bli", 1);
        IEventType event3 = new DummyInteraction("blup", 1);

        // now test with iterations of iterations
        for (int i = 0; i < 5; i++) {
            for (int j = 0; j < 5; j++) {
                simulateEvent(event1, eventTarget1);
            }
            for (int j = 0; j < 5; j++) {
                simulateEvent(event2, eventTarget1);
            }
            for (int j = 0; j < 5; j++) {
                simulateEvent(event3, eventTarget1);
            }
        }

        new TaskTreeChecker(true).assertTaskInstanceList
            ("UserSession session {" +
             "  Iteration iteration2 {" +
             "    Sequence sequence1 {" +
             "      Iteration iteration3 {" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "      }" +
             "      Iteration iteration4 {" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "      }" +
             "    }" +
             "    Sequence sequence1 {" +
             "      Iteration iteration3 {" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "      }" +
             "      Iteration iteration4 {" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "      }" +
             "    }" +
             "    Sequence sequence1 {" +
             "      Iteration iteration3 {" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "      }" +
             "      Iteration iteration4 {" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "      }" +
             "    }" +
             "    Sequence sequence1 {" +
             "      Iteration iteration3 {" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "      }" +
             "      Iteration iteration4 {" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "      }" +
             "    }" +
             "    Sequence sequence1 {" +
             "      Iteration iteration3 {" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "        Event bla {}" +
             "      }" +
             "      Iteration iteration4 {" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "        Event bli {}" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "        Event blup {}" +
             "      }" +
             "    }" +
             "  }" +
             "}", manager.getTaskModel().getUserSessions().get(0));

    }

    /**
     *
     */
    @Test
    public void testSequenceIterationDetection_01() throws Exception {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventType event1 = new DummyInteraction("bla", 1);
        IEventType event2 = new DummyInteraction("bli", 1);
        IEventType event3 = new DummyInteraction("blup", 1);

        simulateEvent(event1, eventTarget1);
        simulateEvent(event2, eventTarget1);
        simulateEvent(event3, eventTarget1);

        simulateEvent(event1, eventTarget1);
        simulateEvent(event2, eventTarget1);
        simulateEvent(event3, eventTarget1);
        
        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Iteration iteration1 {" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "  }" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    @Test
    public void testSequenceIterationDetection_02() throws Exception {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventType event1 = new DummyInteraction("bla", 1);
        IEventType event2 = new DummyInteraction("bli", 1);
        IEventType event3 = new DummyInteraction("blup", 1);

        for (int i = 0; i < 10; i++) {
            simulateEvent(event1, eventTarget1);
            simulateEvent(event2, eventTarget1);
            simulateEvent(event3, eventTarget1);
        }

        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Iteration iteration1 {" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence2 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "  }" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    @Test
    public void testSequenceIterationDetection_03() throws Exception {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventType event1 = new DummyInteraction("bla", 1);
        IEventType event2 = new DummyInteraction("bli", 1);
        IEventType event3 = new DummyInteraction("blup", 1);
        
        // now test with preceding and trailing other interactions
        IEventType event4 = new DummyInteraction("ble", 1);
        IEventType event5 = new DummyInteraction("blo", 1);
        IEventType event6 = new DummyInteraction("blu", 1);
        
        simulateEvent(event4, eventTarget1);
        for (int i = 0; i < 5; i++) {
            simulateEvent(event1, eventTarget1);
            simulateEvent(event2, eventTarget1);
            simulateEvent(event3, eventTarget1);
        }
        simulateEvent(event5, eventTarget1);
        simulateEvent(event6, eventTarget1);

        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Event ble {}" +
             "  Iteration iteration2 {" +
             "    Sequence sequence3 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence3 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence3 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence3 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "    Sequence sequence3 {" +
             "      Event bla {}" +
             "      Event bli {}" +
             "      Event blup {}" +
             "    }" +
             "  }" +
             "  Event blo {}" +
             "  Event blu {}" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    @Test
    public void testSequenceIterationDetection_04() throws Exception {
        IEventTarget eventTarget1 = new DummyGUIElement("elem1");
        IEventType event1 = new DummyInteraction("bla", 1);
        IEventType event2 = new DummyInteraction("bli", 1);
        IEventType event3 = new DummyInteraction("blup", 1);
        IEventType event4 = new DummyInteraction("blo", 1);
        IEventType event5 = new DummyInteraction("blau", 1);
        IEventType event6 = new DummyInteraction("blass", 1);
        IEventType event7 = new DummyInteraction("ble", 1);
        IEventType event8 = new DummyInteraction("blum", 1);
        IEventType event9 = new DummyInteraction("blop", 1);

        // now test with iterations of iterations
        for (int i = 0; i < 5; i++) {
            for (int j = 0; j < 5; j++) {
                simulateEvent(event1, eventTarget1);
                simulateEvent(event2, eventTarget1);
                simulateEvent(event3, eventTarget1);
            }
            for (int j = 0; j < 5; j++) {
                simulateEvent(event4, eventTarget1);
                simulateEvent(event5, eventTarget1);
                simulateEvent(event6, eventTarget1);
            }
            for (int j = 0; j < 5; j++) {
                simulateEvent(event7, eventTarget1);
                simulateEvent(event8, eventTarget1);
                simulateEvent(event9, eventTarget1);
            }
        }

        new TaskTreeChecker().assertTaskInstanceList
            ("UserSession session {" +
             "  Iteration iteration3 {" +
             "    Sequence sequence1 {" +
             "      Iteration iteration4 {" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "      }" +
             "      Iteration iteration6 {" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "      }" +
             "    }" +
             "    Sequence sequence1 {" +
             "      Iteration iteration4 {" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "      }" +
             "      Iteration iteration6 {" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "      }" +
             "    }" +
             "    Sequence sequence1 {" +
             "      Iteration iteration4 {" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "      }" +
             "      Iteration iteration6 {" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "      }" +
             "    }" +
             "    Sequence sequence1 {" +
             "      Iteration iteration4 {" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "      }" +
             "      Iteration iteration6 {" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "      }" +
             "    }" +
             "    Sequence sequence1 {" +
             "      Iteration iteration4 {" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "        Sequence sequence4 {" +
             "          Event bla {}" +
             "          Event bli {}" +
             "          Event blup {}" +
             "        }" +
             "      }" +
             "      Iteration iteration5 {" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "        Sequence sequence5 {" +
             "          Event blo {}" +
             "          Event blau {}" +
             "          Event blass {}" +
             "        }" +
             "      }" +
             "      Iteration iteration6 {" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "        Sequence sequence6 {" +
             "          Event ble {}" +
             "          Event blum {}" +
             "          Event blop {}" +
             "        }" +
             "      }" +
             "    }" +
             "  }" +
             "}", manager.getTaskModel().getUserSessions().get(0));
    }

    /**
     *
     */
    private void simulateEvent(IEventType eventType, IEventTarget eventTarget) {
        manager.handleNewEvent(new Event(eventType, eventTarget));
    }

}
