//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescription;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemSeverityLevel;

/**
 * <p>
 * The result of the usability evaluation, which contains all detected problems.
 * </p>
 * 
 * @author Alexander Deicke
 */
public class UsabilityResult {

    /**
     * <p>
     * All detected problems and their appropriate severity level.
     * </p>
     */
    private Multimap<UsabilityProblemSeverityLevel, UsabilityProblemDescription> detectedProblems;

    /**
     * 
     * <p>
     * Constructor. Creates a new result.
     * </p>
     * 
     */
    public UsabilityResult() {
        this.detectedProblems = ArrayListMultimap.create();
    }

    /**
     * <p>
     * Adds a usability problem to the result.
     * </p>
     * 
     * @param usabilityProblem
     *            the problem, which should be added.
     */
    public void addProblem(UsabilityProblemDescription usabilityProblem) {
        this.detectedProblems.put(usabilityProblem.getSeverityLevel(), usabilityProblem);
    }

    /**
     * 
     * <p>
     * Checks, if problems were found during the usability evaluation.
     * </p>
     * 
     * @return true, iff problems were found
     */
    public boolean hasDetectedProblems() {
        return !this.detectedProblems.isEmpty();
    }

}
