//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usageprofiles;

import java.util.List;

/**
 * <p>
 * A trie processor can be used to recursively process a trie. It will be called for each path
 * stored in the trie. The method called is the {@link #process(List, int)} method. It is provided
 * with the symbol path and the number of its occurrences. The processor may decide for each
 * call to the method using its return value, if the processing shall continue, if the path and all
 * its subsequent symbols can be skipped, or if the whole processing shall be broken up.
 * </p>
 * 
 * @author Patrick Harms
 */
public interface TrieProcessor<T> {
    
    /**
     * <p>
     * potential results for a call to the {@link TrieProcessor#process(List, int)} method. The
     * processor may decide for each call to the method using its return value, if the processing
     * shall continue, if the path and all its subsequent symbols can be skipped, or if the whole
     * processing shall be broken up.
     * </p>
     */
    public enum Result {
        CONTINUE, SKIP_NODE, BREAK
    }

    /**
     * <p>
     * called for each path of symbols stored in the trie which is processed. The method is
     * provided with the symbol path and the number of its occurrences. The processor may decide
     * for each call to the method using its return value, if the processing shall continue, if
     * the path and all its subsequent symbols can be skipped, or if the whole processing shall
     * be broken up.
     * </p>
     * 
     * @param sequence the symbol path to be processed
     * @param count    the number of occurrences of the provided symbol path in the trained
     *                 sequence
     *                 
     * @return as described
     */
    public Result process(List<T> sequence, int count);
    
}
