//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.guitar;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.plugin.guitar.eventcore.GUITAREventTarget;
import de.ugoe.cs.autoquest.plugin.guitar.eventcore.GUITAREventType;
import de.ugoe.cs.autoquest.plugin.guitar.eventcore.GUITARReplayable;
import de.ugoe.cs.autoquest.usageprofiles.DeterministicFiniteAutomaton;
import de.ugoe.cs.autoquest.usageprofiles.FirstOrderMarkovModel;
import edu.umd.cs.guitar.model.GUITARConstants;
import edu.umd.cs.guitar.model.IO;
import edu.umd.cs.guitar.model.data.EFG;
import edu.umd.cs.guitar.model.data.EventGraphType;
import edu.umd.cs.guitar.model.data.EventType;

/**
 * <p>
 * Provides functionality to generates models defined in the package
 * de.ugoe.cs.autoquest.usageprofiles from EFGs.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class EFGModelGenerator {

	/**
	 * <p>
	 * Generates a {@link FirstOrderMarkovModel} from an EFG. In the generated
	 * model, all following events are equally possible, i.e., the model is
	 * equal to a {@link DeterministicFiniteAutomaton}. 
	 * </p>
	 * 
	 * @param efgFileName
	 *            name of the EFG file
	 * @return model generated from the EFG
	 */
	public FirstOrderMarkovModel efgToFirstOrderMarkovModel(String efgFileName) {
		EFG efg = (EFG) IO.readObjFromFile(efgFileName, EFG.class);

		Collection<List<Event>> subsequences = generateEdgeSequences(efg);
		FirstOrderMarkovModel model = new FirstOrderMarkovModel(new Random());
		model.train(subsequences);
		return model;
	}

	/**
	 * <p>
	 * Generates a {@link DeterministicFiniteAutomaton} from an EFG.
	 * </p>
	 * 
	 * @param efgFileName
	 *            name of the EFG file
	 * @return model generated from the EFG
	 */
	public DeterministicFiniteAutomaton efgToDeterministicFiniteAutomaton(
			String efgFileName) {
		EFG efg = (EFG) IO.readObjFromFile(efgFileName, EFG.class);

		Collection<List<Event>> subsequences = generateEdgeSequences(efg);
		DeterministicFiniteAutomaton model = new DeterministicFiniteAutomaton(
				new Random());
		model.train(subsequences);
		return model;
	}

	/**
	 * <p>
	 * Extracts the graph structure from the EFG. The result is a set of
	 * sequences, where each sequence has length two and represents an edge in
	 * the EFG.
	 * </p>
	 * 
	 * @param efg
	 *            EFG for which the edge sequence set is generated
	 * @return edge sequence set
	 */
	private Collection<List<Event>> generateEdgeSequences(EFG efg) {
		List<Event> events = createEvents(efg);
		/*
		 * getEventGraph returns an adjacency matrix, i.e., a square matrix of
		 * efgEvents.size(), where a 1 in row i, column j means an edge
		 * efgEvents.get(i)->efgEvents.get(j) exists.
		 */
		EventGraphType efgGraph = efg.getEventGraph();
		Collection<List<Event>> subsequences = new LinkedList<List<Event>>();

		int efgSize = events.size();
		for (int row = 0; row < efgSize; row++) {
			for (int col = 0; col < efgSize; col++) {
				int relation = efgGraph.getRow().get(row).getE().get(col);
				// otherEvent is followed by currentEvent
				if (relation != GUITARConstants.NO_EDGE) {
					List<Event> edge = new LinkedList<Event>();
					edge.add(events.get(row));
					edge.add(events.get(col));
					subsequences.add(edge);
				}
			}
		}
		return subsequences;
	}

	/**
	 * <p>
	 * Extracts creates {@link EFGEvent} for every event contained in the EFG.
	 * </p>
	 * 
	 * @param efg
	 *            EFG for which the events are created
	 * @return list of events
	 */
	private List<Event> createEvents(EFG efg) {
		List<EventType> efgEvents = efg.getEvents().getEvent();
		List<Event> myEvents = new ArrayList<Event>(efgEvents.size());
		for (EventType event : efgEvents) {
			/*
			 * the widgetId and eventId are only hash values, the
			 * "interpretation" is found in the GUI file.
			 */
		    GUITAREventType type = new GUITAREventType(event.getEventId());
		    GUITAREventTarget target = new GUITAREventTarget(event.getWidgetId());
		    Event myEvent = new Event(type, target);
		    myEvent.addReplayable(new GUITARReplayable(event.getEventId()));
		}
		return myEvents;
	}
}
