//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability.rules.metrics;

import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.absent;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.highRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.infoRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.lowRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.mediumRecommendationSeverityLevel;
import static de.ugoe.cs.autoquest.usability.testutil.FestConditionUtil.present;
import static org.fest.assertions.api.Assertions.assertThat;

import org.junit.Test;

import com.google.common.base.Optional;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.usability.result.UsabilityProblemDescription;
import de.ugoe.cs.autoquest.usability.testutil.GenerateTaskModelUtil;

/**
 * <p>
 * TODO comment
 * </p>
 * 
 * @author Alexander Deicke
 */
public class TextInputRatioEvaluatorTest {

    @Test
    public void should_return_no_recommendation() {
        // Given
        String spec = "UserSession {" +
        			  "  Sequence seq1 {" + 
        			  "    Interaction iter1 {}" + 
        			  "    TextInput t1 (a) {}" + 
        			  "    Interaction iter1 {}" + 
        			  "    Interaction t2 {}" + 
        			  "    TextInput t3 (c) {}" + 
        			  "    Interaction t4 {}" + 
        			  "    Interaction t5 {}" + 
        			  "    Interaction t6 {}" + 
        			  "  }" +
        			  "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(absent());
    }

    @Test
    public void should_return_recommendation_with_info_severity_level() {
        // Given
        String spec = "UserSession {" +
        			  "  Sequence se1 {" + 
        			  "    Interaction t1 {}" + 
        			  "    TextInput t2 (a) {}" + 
        			  "    Interaction t3 {}" + 
        			  "    Interaction t4 {}" + 
        			  "    TextInput t5 (c) {}" +
        			  "  }" + 
        			  "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(infoRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_low_severity_level() {
        // Given
        String spec = "UserSession {" +
        			  "  Sequence seq1 {" + 
        			  "    Interaction t1 {}" + 
        			  "    TextInput t2 (a) {}" + 
        			  "    TextInput t3 (b) {}" + 
        			  "    Interaction t4 {}" + 
        			  "    TextInput t5 (c) {}" + 
        			  "  }" +
        			  "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(lowRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_medium_severity_level() {
        // Given
        String spec = "UserSession {" +
        			  "  Sequence seq1 {" + 
        			  "    TextInput t1 (a) {}" + 
        			  "    TextInput t2 (b) {}" + 
        			  "    Interaction t3 {}" + 
        			  "    TextInput t4 (c) {}" + 
        			  "  }" +
        			  "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(mediumRecommendationSeverityLevel());
    }

    @Test
    public void should_return_recommendation_with_high_severity_level() {
        // Given
        String spec = "UserSession {" +
        			  "  TextInput t1 (bla) {}" +
        			  "}";
        ITaskModel taskTree = GenerateTaskModelUtil.getTaskModelFromSpec(spec);
        // When
        Optional<UsabilityProblemDescription> recommendation = new TextInputRatioMetric(taskTree).calculate();
        // Then
        assertThat(recommendation).is(present()).has(highRecommendationSeverityLevel());
    }
}
